'''
#################################################################################
#                                                                               #
#    Convert Library                                                            #
#                                                                               #
#################################################################################
# Copyright 2022 Kai Masemann                                                   #
#################################################################################
# This file is part of Lonnox CUT                                               #
#                                                                               #
#   Lonnox CUT is free software: you can redistribute it and/or modify          #
#   it under the terms of the GNU General Public License as published by        #
#   the Free Software Foundation, either version 3 of the License, or           #
#   (at your option) any later version.                                         #
#                                                                               #
#   Lonnox CUT is distributed in the hope that it will be useful,               #
#   but WITHOUT ANY WARRANTY; without even the implied warranty of              #
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               #
#   GNU General Public License for more details.                                #
#                                                                               #
#   You should have received a copy of the GNU General Public License           #
#   along with Lonnox CUT.  If not, see <http://www.gnu.org/licenses/>.         #
#                                                                               #
#################################################################################
#	C O N T E N T	     					                                    #
#-------------------------------------------------------------------------------#
#                                                                               #
#    00.00 Load Librarys                                                        #
#                                                                               #
#    01.00 Global Variables                                                     #
#                                                                               #
#    02.00 TrunCAD dxf (Masterwood) import                                      #  
#                                                                               #
#    03.00 Potrace SVG import                                                   #  
#                                                                               #
#    04.00 Add New Job                                                          #  
#                                                                               #
#################################################################################
'''
__version__ = '1.0'
__license__ = "license.txt"
__author__ = 'Kai Masemann <info@lonnox.de>'

#################################################################################
#                                                                               #
#    00.00 Load Libraries                                                       #
#                                                                               #
#################################################################################

#---Libraries For The Layout---
import math
import os
import sys
import uni
import gcode
import subprocess
from PyQt5.QtGui import QPixmap 
from PyQt5.QtGui import QPainter 
from PyQt5.QtGui import QColor 


#################################################################################
#                                                                               #
#    01.00 Gloabl Variables                                                     #
#                                                                               #
#################################################################################

#---File And Dirnames for frozen/ unfrozen application---
if getattr(sys, 'frozen', False):
    # frozen
    path = os.path.dirname(sys.executable) + "/"
else:
    # unfrozen
    path = os.path.dirname(os.path.realpath(__file__)) + "/"
homePath = os.path.expanduser("~") + "/LonnoxCUT/"



#############################################################################
#                                                                           #
#    02.00 TrunCAD dxf (Masterwood) import                                  #
#                                                                           #
#############################################################################
def truncad( self, joblist, filepath, offset ):
    L = uni.language
    #---read data from file---
    jobs=[]; job=[]; x=[]; y=[]; z=[]; 
    with open( filepath, encoding="utf-8") as file:
        data = file.read().split("\n")
        r = d = b = 0
        lt = pt = c = l = sq = ""             

        #---extract entities---
        #dxfEntity = []
        #for i,value in enumerate(data):
        #    if not i%2 and value=="0": dxfEntity.append(["0"])
        #    else: dxfEntity[-1].append(value)
        #print("dxf")        
        #[print(row) for row in dxfEntity]

        #---assign group codes to variables---
        bPrev = 0.0; vDrills = []; lDrills = []
        for i in range(len(data)-2):
            if i%2: continue 
            #---Group Code Assignment---
            if   data[i]=="0" and data[i+1]=="SEQEND": sq = ""; e  = data[i+1] #Sequence End
            elif data[i]=="0"                        : e  = data[i+1] #Entity
            elif data[i]=="6"                        : lt = data[i+1] #Line Type           
            elif data[i]=="8"                        : l  = data[i+1] #Layer
            elif data[i]=="10"                       : x.append( float(data[i+1]) ) #X           
            elif data[i]=="20"                       : y.append( float(data[i+1]) ) #Y           
            elif data[i]=="30"                       : z.append( float(data[i+1]) ) #Z           
            elif data[i]=="39"                       : d = float(data[i+1]) #Depth           
            elif data[i]=="40"                       : r = float(data[i+1]) #Radius           
            elif data[i]=="42" and e=="VERTEX"       : b = float(data[i+1]) #Bow            
            elif data[i]=="62"                       : c  = data[i+1] #Color z.B." 5 "          
            elif data[i]=="66" and data[i+1]=="1"    : sq = e         #Sequence Start            
            elif data[i]=="70"                       : pt = data[i+1] #Polyline Type  
             
            
            #---Entity End Actions---
            if data[i+2]=="0":
                #---Rawpart---
                #(ZeroX,ZeroY,Orientation,X,Y,Width,Height,Thickness)
                if sq==e=="POLYLINE" and l=="PANEL": 
                    rawXMin=float('inf'); rawYMin=float('inf'); rawXMax=0; rawYMax=0; rawT=d
                if sq=="POLYLINE" and l=="PANEL" and e=="VERTEX":
                    rawXMin = min(rawXMin,x[-1])
                    rawYMin = min(rawYMin,y[-1])
                    rawXMax = max(rawXMax,x[-1])
                    rawYMax = max(rawYMax,y[-1])
                if sq=="" and e=="SEQEND" and l=="PANEL":   
                    job = [[0,0,[0]*18],"Rawpart",offset,offset,"Normal",
                           rawXMin,rawYMin,rawXMax-rawXMin,rawYMax-rawYMin,rawT]
                    addJob( joblist, jobs, job)
                    x=[]; y=[]; d=0; l=""

                #---Contour---           
                #CGroup(ZeroX,ZeroY,Orientation,ContourType,Cutpath,Depth,Cutdepth,Feed)
                if sq==e=="POLYLINE" and l[:7]=="Contour":
                    p = gcode.dxfExtract( self, l) 
                    job = [[0,0,[0]*18],"Toolchange","","",p["tool"],p["tool"]]
                    addJob( joblist, jobs, job ) 
                    job = [[0,+1,[0]*18],"Contour-Group",offset,offset,"Normal",
                           "Open",p["cutpath"],math.fabs(d),p["cutdepth"],p["feed"]]
                    cIndex = len(jobs)       
                    addJob( joblist, jobs, job ) 
                #CPoint(ZeroX,ZeroY,X,Y,Radius)
                if sq=="POLYLINE" and l[:7]=="Contour" and e=="VERTEX":
                    #---check if point already exist---
                    points = [] #p[0] = polyline info
                    for i in range(len(x)-1): points.append( [x[i],y[i]] ) 
                    if not [x[-1],y[-1]] in points[1:]: 
                        #---convert dxf radius to contour radius---
                        if bPrev != 0.0: 
                            for bx,by,br in gcode.dxfContourRadius( self, (x[-2],y[-2]),(x[-1],y[-1]),bPrev): 
                                job = [[0,0],"Contour-Point",0,0,bx,by,br]
                                addJob( joblist, jobs, job ) 
                        #---add dxf points to contour---
                        bPrev = b
                        job = [[0,0,[0]*18],"Contour-Point",0,0,x[-1],y[-1],0.0]
                        addJob( joblist, jobs, job )
                #CEnd("","")
                if sq=="" and e=="SEQEND" and l[:7]=="Contour":
                    job = [[0,-1,[0]*18],"Contour-End","",""]
                    addJob( joblist, jobs, job ) 
                    if [x[1],y[1]] == [x[-1],y[-1]]: jobs[cIndex][5] = "Close"
                    x=[]; y=[]; d = b = 0; l="" 

                #---hole line 5mm/vertical drill---
                #line5mm(ZeroX,ZeroY,Orientation,Xa,Xb,Y1,Y2,Y3,Y4,Y5,Y6,Depth,Plunge feed)
                #vdrill(ZeroX,ZeroY,Orientation,X,Y,Depth,Cutdepth,Plunge feed)                
                if sq=="" and e=="CIRCLE" and l=="TOP":    
                    if r == 2.5: lDrills.append( [r,y[-1],x[-1],d] )
                    else: vDrills.append( [r,y[-1],x[-1],d] )
                    x=[]; y=[]; d = r = 0; l=""
                if data[i+3]=="EOF":
                    #---compare every line drills position...---
                    lDs = sorted(lDrills)
                    for i in range(0,len(lDs)-1):
                        if not lDs[i][0]: continue
                        xPos = 32.00
                        lDIs = []
                        #---...with every following line drills position to find 32mm line holes---
                        for j in range(i+1,len(lDs)):
                            if not lDs[j][0]: continue
                            #---collect holes in a line---
                            if lDs[i][1]==lDs[j][1] and lDs[i][3]==lDs[j][3] and (lDs[i][2]+xPos)==lDs[j][2]:
                                lDIs.append(j)
                                xPos += 32.00
                        #---check minimum holeline length related to bitAmount---        
                        if len(lDIs)+1 < uni.settings[7]: continue
                        #---create and add holeline job---
                        job=[[0,0,[0]*18],"HoleLine-5mm",offset,offset,"Normal",
                             lDs[i][2],lDs[lDIs[-1]][2],lDs[i][1],0,0,0,0,0,lDs[i][3],3000]
                        for lDI in lDIs: lDs[lDI][0] = 0           
                        lDs[i][0] = 0
                        addJob( joblist, jobs, job ) 
                    #---add jobs for vertical drills---  
                    vDs = sorted( vDrills + [lD for lD in lDs if lD[0]!=0] )                      
                    lastTool = "" 
                    for vr,vy,vx,vd in vDs:
                        #---toolchange---
                        if lastTool != vr:
                            job = [[0,0,[0]*18],"Toolchange","","","0","? - Radius: "+str(vr)+"mm"]
                            toolUnknown = True
                            for no,text,on,off,m6,g43,dia,cdir,typ,ico in uni.toollist:
                                try: vrComp = float(dia)
                                except ValueError: vrComp = "" 
                                if "z-" in cdir and typ=="Drill bit" and vrComp==vr: 
                                    job = [[0,0,[0]*18],"Toolchange","","",no,no+" - "+text]
                                    toolUnknown = False
                                    break
                            addJob( joblist, jobs, job ) 
                            lastTool = vr                                
                        #---add vertical drill job---                            
                        job = [[toolUnknown,0,[0]*18],"VerticalDrill",offset,offset,"Normal",vx,vy,vd,15.00,3000]
                        addJob( joblist, jobs, job )
                        
                #---horizontal drill---
                #("ZeroX","ZeroY","Orientation","Direction","X","Y1",...,"Y6","Z","Depth","Plunge feed")
                #hdrill are drawn as rectangle in dxf, width = drill size, length = drill depth
                if sq==e=="POLYLINE" and (l=="RIGHT" or l=="LEFT"):
                    hdXMin=float('inf'); hdYMin=float('inf'); hdXMax=0; hdYMax=0; hdZ=rawT-d
                if sq=="POLYLINE" and e=="VERTEX" and (l=="RIGHT" or l=="LEFT"):
                    #---Get the outline dimensions of the hdrill rectangle--- 
                    hdXMin = min(hdXMin,x[-1])
                    hdYMin = min(hdYMin,y[-1])
                    hdXMax = max(hdXMax,x[-1])
                    hdYMax = max(hdYMax,y[-1])
                    #print("A",x,y)
                if sq=="" and e=="SEQEND" and (l=="RIGHT" or l=="LEFT"):
                    if l=="RIGHT": hdDir="X negative"; hdX = hdXMax
                    else: hdDir="X positive"; hdX = hdXMin
                    job = [[0,0,[0]*18],"HorizontalDrill",offset,offset,"Normal",
                           hdDir,hdX,round((hdYMin+hdYMax)/2,2),0,0,0,0,0,hdZ,hdXMax-hdXMin,3000]
                    addJob( joblist, jobs, job)
                    #x=[]; y=[]; d=0; l=""

                #---Groovesaw X---
                #("ZeroX","ZeroY","Orientation","Direction","X1","X2","Y","Depth","Width","blade","Feed")
                if sq==e=="POLYLINE" and l=="SAW":
                    sawXMin=float('inf'); sawYMin=float('inf'); sawXMax=0; sawYMax=0; sawD=d
                if sq=="POLYLINE" and e=="VERTEX" and l=="SAW":
                    sawXMin = min(sawXMin,x[-1])
                    sawYMin = min(sawYMin,y[-1])
                    sawYMax = max(sawYMax,y[-1])
                    sawXMax = max(sawXMax,x[-1])
                if sq=="" and e=="SEQEND" and l=="SAW":
                    job = [[0,0,[0]*18],"Groovesaw X",offset,offset,"Normal",
                           "Climb milling",sawXMin,sawXMax,sawYMin,sawD,sawYMax-sawYMin,4,10000]
                    addJob( joblist, jobs, job)
                    x=[]; y=[]; d=0; l=""
        
        #---Convert Number Parameters To String---
        for i,job in enumerate(jobs):
            for j,para in enumerate(job):
                if isinstance(para,float) or isinstance(para,int):
                    jobs[i][j] = "{:.3f}".format( jobs[i][j] )
    return jobs 
    

#############################################################################
#                                                                           #
#    03.00 Potrace SVG Import                                               #
#                                                                           #
#############################################################################
def potrace( self, joblist, filepath, margin ):
    
    #---Validate Path--- 
    if not os.path.exists( filepath ): return [[],[]]

    #---Convert Images To BMP And Paint Alpha White---
    if os.path.exists( homePath+"popic.bmp" ): os.remove( homePath+"popic.bmp" )  
    if os.path.exists( homePath+"popic.svg" ): os.remove( homePath+"popic.svg" )  
    pic = QPixmap( filepath ) 
    self.px = QPixmap( pic.rect().width(),pic.rect().height() )
    self.px.fill( QColor("#ffffff") )  
    pb = QPainter( self.px )
    pb.drawPixmap( 0,0,pic )
    pb.end()
    self.px.save( homePath+"popic.bmp","BMP" )

    #---Create Vector Graphic With PoTrace---
    if sys.platform in ["linux2","linux"]:
        if margin == 1:
            r = subprocess.call(["potrace", homePath+"popic.bmp", "--tight", 
                             "-o", homePath+"popic.svg", "-s"])
        else:                     
            r = subprocess.call(["potrace", homePath+"popic.bmp", 
                             "-o", homePath+"popic.svg", "-s"])
        print("po",r)                     
    elif sys.platform == "win32":
        suinfo = subprocess.STARTUPINFO()
        suinfo.dwFlags |= subprocess.STARTF_USESHOWWINDOW
        if margin == 1:
            subprocess.call([path+"data/potrace.exe", homePath+"popic.bmp", "--tight", "-o", 
                             homePath+"popic.svg", "-s"], startupinfo=suinfo)
        else:     
            subprocess.call([path+"data/potrace.exe", homePath+"popic.bmp", "-o", 
                             homePath+"popic.svg", "-s"], startupinfo=suinfo)
    else: return [[],[]]
        
    #---Read SVG File---
    if not os.path.exists( homePath+"popic.svg" ): return [[],[]]
    with open( homePath+"popic.svg", encoding="utf-8") as svg:
        tags = svg.read().split("<")
        svgSize = {"viewBox" : 0.0 }
        svgPathes = []
        #---Find Tags---
        for tag in tags:
            #---Find ViewBox Tag---
            if tag.startswith("svg "):
                start = tag.find('viewBox="') + 9
                end = tag.find('"',start)
                vb = tag[ start:end ]
                svgSize = {"viewBox" : [float(v)/0.1 for v in vb.split(" ")] }
            #---Find Translate Tag To Remove Margin---    
            if tag.startswith("g "):
                start = tag.find('translate(') + 10
                end = tag.find(')',start)
                tr = tag[ start:end ]
                if margin == 1: 
                    tx,ty = [float(v)/0.1 for v in tr.split(",")]
                    ty = svgSize["viewBox"][3] - ty
                else: tx, ty = [0,0] #disable margin removing
            #---Find Path Tag---           
            if tag.startswith("path "): 
                #---Format Path String---  
                tag = tag.replace('path d="','')
                tag = tag.replace('"/>','')
                tag = tag.replace('\n',' ')
                tag = tag.replace('z','')
                tag = tag.strip()
                tPath = tag.split(" ")
                #---Extract Geometry Values---
                svgPath = []
                gx,gy = [ 0, 0 ] 
                cmd = ""
                i = 0
                while i < len(tPath):
                    #---Use Commands Modal---
                    if tPath[i][:1].isalpha(): 
                        cmd = tPath[i][:1] 
                        tPath[i] = tPath[i][1:]
                    #---Filter Close Contour Cmd---
                    if tPath[i][-1:].isalpha():
                        tPath[i] = tPath[i][-1:]
                    #---Divide Multiple Polygons Inside A Path---
                    if (cmd in ["M","m"]) and (svgPath != []):
                        svgPathes.append(svgPath)
                        svgPath = []                        
                    #---Move To (absolute) / G0 X Y---
                    if cmd == "M": 
                        gx = int(tPath[i]) + tx
                        gy = int(tPath[i+1]) + ty
                        svgPath.append( ["G0",gx,gy] )
                        i += 2
                    #---Move To (relative) / G0 X Y---    
                    elif cmd == "m": 
                        gx = gx + int(tPath[i])
                        gy = gy + int(tPath[i+1])  
                        svgPath.append( ["G0",gx,gy] )
                        i += 2
                    #---Line To (absolute) / G1 X Y---    
                    elif cmd == "L": 
                        gx = int(tPath[i]) + tx
                        gy = int(tPath[i+1]) + ty
                        svgPath.append( ["G1",gx,gy] )
                        i += 2
                    #---Line To (relative) / G1 X Y---    
                    elif cmd == "l": 
                        gx = gx + int(tPath[i])
                        gy = gy + int(tPath[i+1])  
                        svgPath.append( ["G1",gx,gy] )
                        i += 2
                    #---Cubic Bézier To (absolute) / G5 I J P Q X Y---    
                    elif cmd == "C": 
                        gi = int(tPath[i]) - gx 
                        gj = int(tPath[i+1]) - gy
                        gp = int(tPath[i+2]) - int(tPath[i+4])
                        gq = int(tPath[i+3]) - int(tPath[i+5])
                        gx = int(tPath[i+4]) + tx
                        gy = int(tPath[i+5]) + ty
                        svgPath.append( ["G5",gi,gj,gp,gq,gx,gy] )
                        i += 6
                    #---Cubic Bézier To (relative) / G5 I J P Q X Y---    
                    elif cmd == "c": 
                        gi = int(tPath[i])
                        gj = int(tPath[i+1])
                        gp = gx + int(tPath[i+2]) - (gx + int(tPath[i+4])) #start to endpoint incremental calculation
                        gq = gy + int(tPath[i+3]) - (gy + int(tPath[i+5])) #start to endpoint incremental calculation
                        gx = gx + int(tPath[i+4])
                        gy = gy + int(tPath[i+5])
                        svgPath.append( ["G5",gi,gj,gp,gq,gx,gy] ) #cubic bezier -> Cubic b Spline
                        #print("G5",tPath[i:i+5],gi,gj,gp,gq,gx,gy)
                        i += 6
                    #---Don't Hang On Errors---    
                    else: 
                        i += 1
                        print("Not Supported:", i, cmd, tPath[i] )
                    #print(svgPath[-1])
                svgPathes.append( svgPath )                    
            #print("newPath")
    return [svgSize,svgPathes]        

    
    
#############################################################################
#                                                                           #
#    04.00 Add New Job                                                      #
#                                                                           #
#############################################################################
def addJob( joblist, jobs, job ):

    #---Search Next Available Job No---
    jobNames = [job[1] for job in joblist+jobs]
    
    no = 1
    while "{} {:03d}".format( job[1], no) in jobNames: no+=1
    job[1] = "{} {:03d}".format( job[1], no) 
    jobs.append(job)                
    job = []



#
