'''
#################################################################################
#                                                                               #
#    Lonnox CUT - module Circle                                                 #
#                                                                               #
#################################################################################
#	C O N T E N T                                  								#
#-------------------------------------------------------------------------------#
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#    02.00 g-code generator                                                     #
#        02.01 local vars                                                       #
#        02.02 mirror module on x-axis                                          #
#        02.03 startcodes                                                       #
#        02.04 g-code generation for module                                     #
#        02.05 end codes                                                        #
#                                                                               #
#    03.00 determine start-/endtool                                             #
#                                                                               #
#################################################################################
'''
__version__ = '1.0'
__license__ = "license.txt"
__author__ = 'Kai Masemann <info@lonnox.de>'

#################################################################################
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#################################################################################

#---libraries for the layout---
import math
import os
import sys
import gcode
import uni


#################################################################################
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#################################################################################

#---choose/ add section in the module tree---
section    = "Mill 2D"

#---name of the module in the tree---
name       = "Circle"
groupEnd   = ""

#---explain picture of the module--- 
picture    = "Circle.png"

#---explaining text for helpbutton in lonnox cut---
info = "CircleInfo"

#---predefined tools---
predefinedTools = False

#---widgets that are displayed to set the parameters---
# T=Text, N=NumberValues, O=OptionList, F=FileButton, C=CheckBox
widget = ("N","N","O","N","N","N","N","N","O","O","N")

#---parameter of the module---
labels    = ("Zero Point X", "Zero Point Y", "Orientation","X","Y","Radius (r)",
             "Depth","Cutdepth","Cutpath","Last segment","Feed")

#---option presets for listed parameters---
options   = {"Orientation":      ("Normal","Mirror"),
             "Cutpath":          ("Center","Inside","Outside"),
             "Last segment":     ("Y Min","X Max","Y Max","X Min")}



class module( object ):
    #############################################################################
    #                                                                           #
    #    02.00 g-code generator                                                 #
    #                                                                           #
    #############################################################################
    def gcode( self, jTools, joblist, jIndex, preview=0 ):
        global name
        L = uni.language 

        #########################################################################
        #    02.01 local vars                                                   #
        #########################################################################

        #---extrace name, tool and param rows from joblist table---
        jNames = [job[1] for job in joblist]
        jParams = [job[2:] for job in joblist] #for contents, see labels
        
        #---variable values---
        if name in uni.language: LName = uni.language[name]
        else: LName = name
        code1         = "\n(" + ("-"*30) + LName + jNames[jIndex][-4:] + ("-"*30) + ")\n\n"
        code2         = ""
        x0            = jParams[jIndex][0]
        y0            = jParams[jIndex][1]
        orient        = jParams[jIndex][2]
        x             = jParams[jIndex][3] + x0                     
        y             = jParams[jIndex][4] + y0                       
        r             = math.fabs( jParams[jIndex][5] )   
        depth         = math.fabs( jParams[jIndex][6] )
        cutdepth      = math.fabs( jParams[jIndex][7] )
        cutpath       = jParams[jIndex][8]
        lastseg       = jParams[jIndex][9]
        f             = jParams[jIndex][10]                     
        fPlunge       = f * uni.settings[6] / 100         
        mirrorstops   = uni.settings[1]
        zSave         = uni.settings[4]

        #---load tool on,off,m6,g43 and cutterradius---
        csvT = gcode.csvtool( self, jTools[jIndex][0], jTools[jIndex][2] )
        cr   = csvT["cr"]                             
        
        #---load material thickness---
        i = 1
        thick = 0
        while jIndex >= i:
            if jNames[jIndex-i][:-4] == "Rawpart":
                thick = jParams[jIndex-i][7]
                break;
            i += 1

        #---set preview values---
        if preview: depth=1; cutdepth=1

        #---escape if essetial values are missing---
        if not depth or not cutdepth or not r: return code1

        
        #########################################################################
        #    02.02 option and variable preparation                              #
        #########################################################################
         
        #---recalculate mirror values---
        if orient == "Mirror": x = mirrorstops - x

        #---set/toggle cutter compensation value---
        if cutpath == "Inside":
            #---for proper inner path minimum edge radius must be cutter radius---
            if cr > r : r = cr + 0.001
            cr = -cr 
        if cutpath == "Center": cr = 0

        #---calculate circle coords on 45°, 135°, 225° and 315°---
        rp = r+cr
        xp = math.cos( math.radians(45) ) * rp 
        yp = math.sin( math.radians(45) ) * rp 

        #---build gcode for right/ left moving steps---
        if   lastseg == "Y Min": x1=x+xp; y1=y-yp; x2=x-xp; y2=y-yp  
        elif lastseg == "X Max": x1=x+xp; y1=y+yp; x2=x+xp; y2=y-yp  
        elif lastseg == "Y Max": x1=x-xp; y1=y+yp; x2=x+xp; y2=y+yp  
        elif lastseg == "X Min": x1=x-xp; y1=y-yp; x2=x-xp; y2=y+yp  


        #########################################################################
        #    02.03 startcodes                                                   #
        #########################################################################

        #---add startcode and activate tool---
        code1 += gcode.start( self, thick ) 
        code1 += gcode.toolOptimizer( self, jIndex, jTools, csvT, "on", True )

        
        #########################################################################
        #    02.04 g-code generation for module                                 #
        #########################################################################

        #---move to start position---
        code1 += "G0 Z{:.3f}\n".format( thick+zSave )
        code1 += "G0 X{:.3f} Y{:.3f}\n".format( x1, y1 )
        code2 += "G0 Z{:.3f}\n".format( thick+zSave )
        code2 += "G0 X{:.3f} Y{:.3f}\n".format( x2, y2 )
        
        #---loop until final depth was reached---
        dirtoggle = 0 
        currentdepth = 0.00
        while 1:

            #---set next depth level or final depth---
            currentdepth += cutdepth
            if currentdepth > depth: currentdepth = depth

            #---update depth for rectangle cut and last segment cut---
            code1 += "G1 Z{:.3f} F{:.0f}\n".format( thick-currentdepth, fPlunge )
            code2 += "G1 Z{:.3f} F{:.0f}\n".format( thick-currentdepth, fPlunge )
        

            #---build steps for left cut direction and last segment---  
            if not dirtoggle:
                code1 += "G3 X{:.3f} Y{:.3f} R{:.3f} F{:.0f}\n".format( x2, y2, -rp, f )
                code2 += "G3 X{:.3f} Y{:.3f} R{:.3f} F{:.0f}\n".format( x1, y1, +rp, f )

            #---build steps for right cut direction and last segment---  
            else:
                code1 += "G2 X{:.3f} Y{:.3f} R{:.3f} F{:.0f}\n".format( x1, y1, -rp, f )
                code2 += "G2 X{:.3f} Y{:.3f} R{:.3f} F{:.0f}\n".format( x2, y2, +rp, f )
            dirtoggle ^= 1

            #---if final depth was reached then end while loop---
            if currentdepth == depth: break;
            
        #---put rectangle cut and last segment cut together---
        code1 += "\n" + code2 + "\n"
        code1 += "G0 Z" + str(thick+zSave) + "\n\n"
                  

        #########################################################################
        #    02.05 end codes                                                    #
        #########################################################################
        
        #---deactivate tool and reset modal codes--- 
        code1 += gcode.toolOptimizer( self, jIndex, jTools, csvT, "off", True )
        code1 += gcode.end( self )

        return code1 
   

   
    #############################################################################
    #                                                                           #
    #    03.00 determine start-/endtool                                         #
    #                                                                           #
    #############################################################################
    # function returns start/endtool of THIS module for toolOptimizer 
    # "" will be ignored by optimizer 
    # "nT" will force to load no Tool
    def tool( self, joblist, jIndex ):
        
        tool = gcode.findToolchange( self, joblist, jIndex ) 
        return (tool,tool) #(Start, Endtool)

           