'''
#################################################################################
#                                                                               #
#    Lonnox CUT - module Lochreihe-Regal                                        #
#                                                                               #
#################################################################################
#	C O N T E N T                                  								#
#-------------------------------------------------------------------------------#
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#    02.00 g-code generator                                                     #
#        02.01 local vars                                                       #
#        02.02 mirror module on x-axis                                          #
#        02.03 startcodes                                                       #
#        02.04 g-code generation for module                                     #
#        02.05 end codes                                                        #
#                                                                               #
#    03.00 determine start-/endtool                                             #
#                                                                               #
#################################################################################
'''
__version__ = '1.0'
__license__ = "license.txt"
__author__ = 'Kai Masemann <info@lonnox.de>'

#################################################################################
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#################################################################################

#---libraries for the layout---
import math
import os
import sys
import gcode
import uni


#################################################################################
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#################################################################################

#---choose/ add section in the module tree---
section    = "Drill"

#---name of the module in the tree---
name       = "HoleLine-Shelf"
groupEnd   = ""

#---explain picture of the module--- 
picture    = "HoleLineShelf.png"

#---explaining text for helpbutton in lonnox cut---
info = "HLShelfInfo"

#---predefined tools---
predefinedTools = False

#---widgets that are displayed to set the parameters---
# T=Text, N=NumberValues, O=OptionList, F=FileButton, C=CheckBox
widget = ("N","N","O","N","N","N","N","N","N","N","N","N","N","N","N","N","N","N","N","N")

#---parameter of the module---
labels    = ("Zero Point X", "Zero Point Y", "Orientation","X1","X2","X3","X4","X5","X6",
             "X7","X8","Y1","Y2","Y3","Y4","Y5","Y6","Depth","Cutdepth","Plunge feed")
             
#---option presets for listed parameters---
options   = {"Orientation":      ("Normal","Mirror")}



class module( object ):
    #############################################################################
    #                                                                           #
    #    02.00 g-code generator                                                 #
    #                                                                           #
    #############################################################################
    def gcode( self, jTools, joblist, jIndex, preview=0 ):
        global name
        L = uni.language 
        
        #########################################################################
        #    02.01 local vars                                                   #
        #########################################################################

        #---extrace name, tool and param rows from joblist table---
        jNames = [job[1] for job in joblist]
        jParams = [job[2:] for job in joblist] #for contents, see labels
        
        #---variable values---
        if name in uni.language: LName = uni.language[name]
        else: LName = name
        code          = "\n(" + ("-"*30) + LName + jNames[jIndex][-4:] + ("-"*30) + ")\n\n"
        x0            = jParams[jIndex][0]
        y0            = jParams[jIndex][1]
        orient        = jParams[jIndex][2]
        depth         = math.fabs( jParams[jIndex][17] )
        cutdepth      = math.fabs( jParams[jIndex][18] )
        depthfeed     = jParams[jIndex][19] 
        mirrorstops   = uni.settings[1]
        zSave         = uni.settings[4]
        
        #---load tool on,off,m6,g43 and cutterradius---
        csvT = gcode.csvtool( self, jTools[jIndex][0], jTools[jIndex][2] )
        cr   = csvT["cr"]                             
        
        #---load material thickness---
        i = 1
        thick = 0
        while jIndex >= i:
            if jNames[jIndex-i][:-4] == "Rawpart":
                thick = jParams[jIndex-i][7]
                break;
            i += 1
  
        #---only use x values greater then 0---
        xvalues = []
        for i in range(8):
            x = jParams[jIndex][i+3]  
            if x > 0: xvalues.append( x + x0 ) 

        #---only use y values greater then 0---
        yvalues = []
        for i in range(6):
            y = jParams[jIndex][i+11] 
            if y > 0: yvalues.append( y + y0) 

        #---set preview values---
        if preview: depth=1; cutdepth=10
        
        #---escape if essetial values are missing---
        if not depth or not cutdepth: return code

        
        #########################################################################
        #    02.02 mirror module on x-axis                                      #
        #########################################################################

        #---recalculate mirror values---
        if orient == "Mirror":
            for i in range( len(xvalues) ): xvalues[i] = mirrorstops - xvalues[i]


        #########################################################################
        #    02.03 startcodes                                                   #
        #########################################################################

        #---add startcode and activate tool---
        code += gcode.start( self, thick ) 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "on", True )

        
        #########################################################################
        #    02.05 g-code generation for module                                 #
        #########################################################################

        #---loop for every x value---
        for x in xvalues:

            #---loop for every y value---
            for y in yvalues:
                
                #---loop until final depth was reached---
                currentdepth = 0.00
                while currentdepth < depth:

                    #---set next depth level or final depth---
                    currentdepth += cutdepth
                    if currentdepth > depth: currentdepth = depth
            
                    #---start position---
                    code += "G0 Z{:.3f}\n".format( thick+zSave )
                    code += "G0 X{:.3f} Y{:.3f}\n".format( x,y )
                      
                    #---with depthfeed to depth---
                    code += "G1 Z{:.3f} F{:.0f}\n".format( thick - currentdepth,depthfeed )

        #---move z up---
        code += "G0 Z{:.3f}\n\n".format( thick+zSave )

                 
        #########################################################################
        #    02.05 end codes                                                    #
        #########################################################################
        
        #---deactivate tool and reset modal codes--- 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "off", True )
        code += gcode.end( self )

        return code 
   


    #############################################################################
    #                                                                           #
    #    03.00 determine start-/endtool                                         #
    #                                                                           #
    #############################################################################
    # function returns start/endtool of THIS module for toolOptimizer 
    # "" will be ignored by optimizer 
    # "nT" will force to load no Tool
    def tool( self, joblist, jIndex ):
        
        tool = gcode.findToolchange( self, joblist, jIndex ) 
        return (tool,tool) #(Start, Endtool)
