'''
#################################################################################
#                                                                               #
#    Lonnox CUT - module Line                                                   #
#                                                                               #
#################################################################################
#	C O N T E N T                                  								#
#-------------------------------------------------------------------------------#
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#    02.00 g-code generator                                                     #
#        02.01 local vars                                                       #
#        02.02 mirror module on x-axis                                          #
#        02.03 startcodes                                                       #
#        02.04 g-code generation for module                                     #
#        02.05 end codes                                                        #
#                                                                               #
#    03.00 determine start-/endtool                                             #
#                                                                               #
#################################################################################
'''
__version__ = '1.0'
__license__ = "license.txt"
__author__ = 'Kai Masemann <info@lonnox.de>'

#################################################################################
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#################################################################################

#---libraries for the layout---
import math
import os
import sys
import gcode
import uni


#################################################################################
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#################################################################################

#---choose/ add section in the module tree---
section    = "Mill 2D"

#---name of the module in the tree---
name       = "Line"
groupEnd   = ""

#---explain picture of the module--- 
picture    = "Line.png"

#---explaining text for helpbutton in lonnox cut---
info = "LineInfo"

#---predefined tools---
predefinedTools = False

#---widgets that are displayed to set the parameters---
# T=Text, N=NumberValues, O=OptionList, F=FileButton, C=CheckBox
widget = ("N","N","O","N","N","N","N","N","O","N","N","N")

#---parameter of the module---
labels    = ("Zero Point X", "Zero Point Y", "Orientation","X1","Y1",
             "X2","Y2","Rotation","Cutpath","Depth","Cutdepth","Feed")

#---option presets for listed parameters---
options   = {"Orientation":     ("Normal","Mirror"),
             "Cutpath":         ("Center","Left","Right")}

             
             
class module( object ):
    #############################################################################
    #                                                                           #
    #    02.00 g-code generator                                                 #
    #                                                                           #
    #############################################################################
    def gcode( self, jTools, joblist, jIndex, preview=0 ):
        global name
        L = uni.language 
        
        #########################################################################
        #    02.01 local vars                                                   #
        #########################################################################

        #---extrace name, tool and param rows from joblist table---
        jNames = [job[1] for job in joblist]
        jParams = [job[2:] for job in joblist] #for contents, see labels
        
        #---variable values---
        if name in uni.language: LName = uni.language[name]
        else: LName = name
        code          = "\n(" + ("-"*30) + LName + jNames[jIndex][-4:] + ("-"*30) + ")\n\n"
        x0            = jParams[jIndex][0]
        y0            = jParams[jIndex][1]
        orient        = jParams[jIndex][2]
        x1            = jParams[jIndex][3] #+ x0 see rotation calculation                      
        y1            = jParams[jIndex][4] #+ y0 see rotation calculation                     
        x2            = jParams[jIndex][5] #+ x0 see rotation calculation                    
        y2            = jParams[jIndex][6] #+ y0 see rotation calculation                    
        rotation      = jParams[jIndex][7]
        cutpath       = jParams[jIndex][8]
        depth         = math.fabs( jParams[jIndex][9] ) 
        cutdepth      = math.fabs( jParams[jIndex][10] ) 
        f             = jParams[jIndex][11]                     
        fPlunge       = f * uni.settings[6] / 100         
        mirrorstops   = uni.settings[1]
        zSave         = uni.settings[4]

        #---load tool on,off,m6,g43 and cutterradius---
        csvT = gcode.csvtool( self, jTools[jIndex][0], jTools[jIndex][2] )
        cr   = csvT["cr"]                             
        
        #---load material thickness---
        i = 1
        thick = 0
        while jIndex >= i:
            if jNames[jIndex-i][:-4] == "Rawpart":
                thick = jParams[jIndex-i][7]
                break;
            i += 1
        
        #---set preview values---
        if preview: depth=1; cutdepth=1        
        
        #---escape if essetial values are missing---
        if not depth or not cutdepth or (x1==x2 and y1==y2): return code


        #########################################################################
        #    02.02 option and variable preparation                              #
        #########################################################################
         
        #---rotate x/y values---
        r1 = gcode.distancePP( self, (0,0), (x1,y1) )
        r2 = gcode.distancePP( self, (0,0), (x2,y2) )
        a1 = rotation + gcode.pointAngle360( self, 0,0, x1,y1, metric=1 )
        a2 = rotation + gcode.pointAngle360( self, 0,0, x2,y2, metric=1 )
        if a1 > 360: a1 -= 360
        if a2 > 360: a2 -= 360
        x1,y1 = gcode.circlePoint( self, x0, y0, a1, r1 ) 
        x2,y2 = gcode.circlePoint( self, x0, y0, a2, r2 ) 
        

        #---recalculate mirror values---
        if orient == "Mirror": x1 = mirrorstops - x1; x2 = mirrorstops - x2 

        #---move line left/ rigth---
        if cutpath != "Center":
            lineAng = gcode.pointAngle360( self, x1, y1, x2, y2, metric=1 )
            if   cutpath == "Left":
                lineAng += 90
                if lineAng >= 360: lineAng -= 360
            elif cutpath == "Right":
                lineAng -= 90
                if lineAng < 0: lineAng += 360
            P1New = gcode.circlePoint( self, x1, y1, lineAng, cr ) 
            P2New = gcode.circlePoint( self, x2, y2, lineAng, cr ) 
        
            #---save new point coords---
            x1 = P1New[0]        
            y1 = P1New[1]        
            x2 = P2New[0]        
            y2 = P2New[1]        
        
        
        #########################################################################
        #    02.03 startcodes                                                   #
        #########################################################################

        #---add startcode and activate tool---
        code += gcode.start( self, thick ) 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "on", True )

        
        #########################################################################
        #    02.04 g-code generation for module                                 #
        #########################################################################

        xy = [ (x1,y1),(x2,y2) ]         
        
        #---move to start position---
        code += "G0 Z{:.3f}\n".format( thick+zSave )
        code += "G0 X{:.3f} Y{:.3f}\n".format( xy[0][0], xy[0][1] )
        
        #---loop until final depth was reached---
        currentdepth = 0.00
        while currentdepth < depth:

            #---set next depth level or final depth---
            currentdepth += cutdepth
            if currentdepth > depth: currentdepth = depth

            #---move to next depth level---
            code += "G1 Z{:.3f} F{:.0f}\n".format( thick-currentdepth, fPlunge )
            code += "G1 X{:.3f} Y{:.3f} F{:.0f}\n".format( xy[1][0], xy[1][1], f )
                
            #---reverse x values after every depth level---    
            xy.reverse()
            
        #---move z up---
        code += "G0 Z" + str(thick+zSave) + "\n\n"
                  

        #########################################################################
        #    02.05 end codes                                                    #
        #########################################################################
        
        #---deactivate tool and reset modal codes--- 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "off", True )
        code += gcode.end( self )
        
        return code 
 

 
    #############################################################################
    #                                                                           #
    #    03.00 determine start-/endtool                                         #
    #                                                                           #
    #############################################################################
    # function returns start/endtool of THIS module for toolOptimizer 
    # "" will be ignored by optimizer 
    # "nT" will force to load no Tool
    def tool( self, joblist, jIndex ):
        
        tool = gcode.findToolchange( self, joblist, jIndex ) 
        return (tool,tool) #(Start, Endtool)

        