'''
#################################################################################
#                                                                               #
#    Lonnox CUT - module Picture Tracer                                         #
#                                                                               #
#################################################################################
#    C O N T E N T                                                                #
#-------------------------------------------------------------------------------#
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#    02.00 g-code generator                                                     #
#        02.01 local vars                                                       #
#        02.02 mirror module on x-axis                                          #
#        02.03 startcodes                                                       #
#        02.04 g-code generation for module                                     #
#        02.05 end codes                                                        #
#                                                                               #
#    03.00 determine start-/endtool                                             #
#                                                                               #
#################################################################################
'''
__version__ = '1.0'
__license__ = "license.txt"
__author__ = 'Kai Masemann <info@lonnox.de>'

#################################################################################
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#################################################################################

#---libraries for the layout---
import math
import os
import sys
import gcode
import uni
import convert
import copy


#################################################################################
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#################################################################################

#---choose/ add section in the module tree---
section    = "Mill 2D"

#---name of the module in the tree---
name       = "Picture-Tracer"
groupEnd   = ""

#---explain picture of the module--- 
picture    = "PictureTracer.png"

#---explaining text for helpbutton in lonnox cut---
info = "TracerInfo"

#---predefined tools---
predefinedTools = False

#---widgets that are displayed to set the parameters---
# T=Text, N=NumberValues, O=OptionList, F=FileButton, C=CheckBox
widget = ("N","N","O","F","N","N","N","C","N","N","N","N","N")

#---parameter of the module---
labels    = ("Zero Point X", "Zero Point Y", "Orientation","Picturefile","X","Y",
             "Width (a)","Remove margin","Rotation","Depth","Cutdepth","Feed",
             "Interpolation-Quality")

#---option presets for listed parameters---
options   = {"Orientation":     ("Normal","Mirror")}



class module( object ):
    #############################################################################
    #                                                                           #
    #    02.00 g-code generator                                                 #
    #                                                                           #
    #############################################################################
    def gcode( self, jTools, joblist, jIndex, preview=0 ):
        global name
        L = uni.language 

        #########################################################################
        #    02.01 local vars                                                   #
        #########################################################################

        #---extrace name, tool and param rows from joblist table---
        jNames = [job[1] for job in joblist]
        jParams = [job[2:] for job in joblist] #for contents, see labels

        #---variable values---
        if name in uni.language: LName = uni.language[name]
        else: LName = name
        code          = "\n(" + ("-"*30) + LName + jNames[jIndex][-4:] + ("-"*30) + ")\n\n"
        x0            = jParams[jIndex][0]
        y0            = jParams[jIndex][1]
        orient        = jParams[jIndex][2]
        file          = jParams[jIndex][3]
        x             = jParams[jIndex][4] + x0                             
        y             = jParams[jIndex][5] + y0                             
        w             = jParams[jIndex][6]                              
        margin        = jParams[jIndex][7]                              
        rotation      = jParams[jIndex][8]                              
        depth         = math.fabs( jParams[jIndex][9] )
        cutdepth      = math.fabs( jParams[jIndex][10] )
        f             = jParams[jIndex][11]                              
        fPlunge       = f * uni.settings[6] / 100         
        mirrorstops   = uni.settings[1]
        zSave         = uni.settings[4]
        div           = int( jParams[jIndex][12] )

        #---load tool on,off,m6,g43 and cutterradius---
        csvT = gcode.csvtool( self, jTools[jIndex][0], jTools[jIndex][2] )
        cr   = csvT["cr"]                             
        
        #---load material thickness---
        i = 1
        thick = 0
        while jIndex >= i:
            if jNames[jIndex-i][:-4] == "Rawpart":
                thick = jParams[jIndex-i][7]
                break;
            i += 1

        #---set preview values---
        if preview: depth=1; cutdepth=1
        
        #---clamp Interpolation division---
        if div < 1: div = 1

        #---escape if essetial values are missing---
        if not os.path.exists(file) or not w or not depth or not cutdepth: return code
        
        
        #########################################################################
        #    02.02 option and variable preparation                              #
        #########################################################################
        
        #---Convert Potrace-SVG To xy Coords---
        svgSize, svgPathes = convert.potrace( self, joblist, file, margin )
        if svgPathes == []: return code  

        #---Calculate Scale Factor For Potrace SVG---
        fac = (w / svgSize["viewBox"][2])

        #---rotate x/y values---
        if rotation != 0:
            pathes = copy.deepcopy(svgPathes) 
            for j,svgPath in enumerate(svgPathes):
                for i,cmd in enumerate(svgPath):
                    #---change g5 coords to absolute---
                    if cmd[0] == "G5":
                        cmd[1] += pathes[j][i-1][-2]
                        cmd[2] += pathes[j][i-1][-1]
                        cmd[3] += pathes[j][i][-2]
                        cmd[4] += pathes[j][i][-1]
                    #---rotate coords---
                    k = 1
                    while k < len(cmd):
                        r = gcode.distancePP( self, (x,y), (cmd[k], cmd[k+1]) )
                        a = rotation + gcode.pointAngle360( self, x, y, cmd[k], cmd[k+1], metric=1 )
                        if a > 360: a -= 360
                        cmd[k], cmd[k+1] = gcode.circlePoint( self, x, y, a, r ) 
                        k += 2  
                    #---change g5 coords back to relative--- 
                    if cmd[0] == "G5":
                        cmd[1] -= svgPath[i-1][-2]
                        cmd[2] -= svgPath[i-1][-1]
                        cmd[3] -= cmd[5]
                        cmd[4] -= cmd[6]
                
        #---split pathes at longest segment---
        pathes = []
        for svgPath in svgPathes:
            l = []
            lx = 0
            ly = 0
            #---collect approximate length--- 
            for i,cmd in enumerate(svgPath):
                if cmd[0] == "G0": 
                    lx = cmd[1]; ly = cmd[2]
                elif i == 0: continue
                elif cmd[0] == "G1": 
                    l.append( math.sqrt( math.fabs(lx-cmd[1])**2 + math.fabs(ly-cmd[2])**2 ) )                 
                    lx = cmd[1]; ly = cmd[2]
                elif cmd[0] == "G5": 
                    l.append( math.sqrt( math.fabs(lx-cmd[5])**2 + math.fabs(ly-cmd[6])**2 ) )
                    lx = cmd[5]; ly = cmd[6]
            #---Divide at least 15% / 85%---
            ls = sum(l)
            lm = max(l)
            svgPath = svgPath[l.index( lm ):] + svgPath[1:l.index( lm )] 
            i = 1
            while lm < (ls * 0.15):
                lm += l[i]
                i += 1
            pathes.append( svgPath[:i+1] )
            pathes.append( svgPath[i+1:] )            
            pathes[-2].insert( 0,["G0",pathes[-1][-1][-2], pathes[-1][-1][-1]] )
            pathes[-1].insert( 0,["G0",pathes[-2][-1][-2], pathes[-2][-1][-1]] )
        svgPathes = copy.deepcopy(pathes)         
        
        #---change straight g5 to g1--- 
        '''
        for j,svgPath in enumerate(svgPathes):
            for i,cmd in enumerate(svgPath):
                if i == 0: continue
                elif cmd[0] == "G5": 
                    l = svgPath[i-1][-2:] + cmd[-2:] 
                    p1 = [ l[0] + cmd[1], l[1] + cmd[2] ]
                    p2 = [ l[2] + cmd[3], l[3] + cmd[4] ]
                    #dis1, onLine1 = gcode.distanceLP( self,l,p1)          
                    #dis2, onLine2 = gcode.distanceLP( self,l,p2)          
                    #if (dis1 < 0.02) and (dis2 < 0.01) and onLine1 and onLine2: 
                    #    svgPathes[j][i] = ["G1",cmd[5],cmd[6]] 
                    a1 = gcode.pointAngle360(self, l[0], l[1], l[2], l[3] ) 
                    a2 = gcode.pointAngle360(self, l[0], l[1], p1[0], p1[1] ) 
                    ad1 = min( [math.fabs(a1-a2),math.fabs((a1-a2)-360)] ) 
                    a3 = gcode.pointAngle360(self, l[2], l[3], l[0], l[1] ) 
                    a4 = gcode.pointAngle360(self, l[2], l[3], p2[0], p2[1] ) 
                    ad2 = min( [math.fabs(a3-a4),math.fabs((a3-a4)-360)] ) 
                    if (ad1 < 0.5) and (ad2 < 0.5): 
                        svgPathes[j][i] = ["G1",cmd[5],cmd[6]] 
        '''

        #---Change G5 to G1 (workaround for LinuxCNC bug)---
        pAmount = 0
        svgPathesI = []
        for j,svgPath in enumerate(svgPathes):
            p0 = [0,0] 
            svgPathI = []
            for i,cmd in enumerate(svgPath):
                #---Interpolate G5 To G1---
                if cmd[0] == "G5": 
                    #["G5",gi,gj,gp,gq,gx,gy]
                    #p0 Start, p1 ControlS, p2 ControlE, p3 End
                    p1 = [p0[0]+cmd[1],p0[1]+cmd[2]]
                    p3 = [cmd[5],cmd[6]]
                    p2 = [p3[0]+cmd[3],p3[1]+cmd[4]]
                    for i in range(div):
                        t = (1/div) * (i+1)
                        xP = ((1-t)**3 * p0[0]) + (3 * t * (1-t)**2 * p1[0]) + (3 * t**2 * (1-t) * p2[0]) + (t**3 * p3[0])
                        yP = ((1-t)**3 * p0[1]) + (3 * t * (1-t)**2 * p1[1]) + (3 * t**2 * (1-t) * p2[1]) + (t**3 * p3[1]) 
                        svgPathI.append( ["G1",xP,yP] )
                        pAmount += 1
                    p0 = p3                    
                #---Save Last Position And Keep Existing G0 And G1 Commands---        
                else:     
                    p0 = [cmd[1],cmd[2]]                    
                    svgPathI.append( cmd )
            svgPathesI.append(svgPathI)        
        svgPathes = svgPathesI      
        
        
        #---mirror the circle segment---
        if orient == "Mirror":
            for svgPath in svgPathes: 
                for cmd in svgPath:        
                    cmd[1] = mirrorstops-cmd[1]


        #########################################################################
        #    02.03 startcodes                                                   #
        #########################################################################

        #---add startcode and activate tool---
        code += gcode.start( self, thick ) 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "on", True )


        #########################################################################
        #    02.04 g-code generation for module                                 #
        #########################################################################

        #---loop for every path, start with inner---
        for svgPath in reversed(svgPathes):
            #---move to start position---
            code += "G0 Z{:.3f}\n".format( thick+zSave )
            if svgPath[0][0] == "G0":
                code += "G0 X{:.3f} Y{:.3f}\n".format( x + (svgPath[0][1] * fac), y + (svgPath[0][2] * fac) )
            else: 
                code += "G0 X{:.3f} Y{:.3f}\n".format( x, y )
            
            #---loop until final depth was reached---
            currentdepth = 0.00
            cutdir = 0
            while 1:
     
                #---set next depth level or final depth---
                currentdepth += cutdepth
                if currentdepth > depth: currentdepth = depth

                #---update depth for rectangle cut and last segment cut---
                code += "G1 Z{:.3f} F{:.0f}\n".format( thick-currentdepth, fPlunge )
                code += "F{:.0f}\n".format( f )
            
                #---set path direction---
                if not cutdir: sP = copy.deepcopy(svgPath)
                else: 
                    sP = copy.deepcopy(svgPath)
                    sP.reverse()
                    sPl = len(sP)-1
                    for i,cmd in enumerate(sP):
                        if i == 0: sPx = cmd[-2]; sPy = cmd[-1]
                        if i < sPl: 
                            sP[i][-2] = sP[i+1][-2]
                            sP[i][-1] = sP[i+1][-1]
                        else:    
                            sP[i][-2] = sPx
                            sP[i][-1] = sPy
                        if cmd[0] == "G5": 
                            sP[i] = [ cmd[0],cmd[3],cmd[4],cmd[1],cmd[2],sP[i][5],sP[i][6] ]                        
                        
                for cmd in sP:
                    #---Add G1 Straight Line---
                    if cmd[0] == "G1":
                        code += "G1 X{:.3f} Y{:.3f}\n".format( x+(cmd[1] * fac), y+(cmd[2] * fac) )
                    #---Add G5 Cubic Bezier---
                    elif cmd[0] == "G5":
                        gi = cmd[1] * fac
                        gj = cmd[2] * fac
                        gp = cmd[3] * fac
                        gq = cmd[4] * fac
                        gx = x + (cmd[5] * fac)
                        gy = y + (cmd[6] * fac)
                        code += "G5 I{:.3f} J{:.3f} P{:.3f} Q{:.3f} X{:.3f} Y{:.3f}\n".format( gi,gj,gp,gq,gx,gy )
                
                #---change direction---
                if cutdir == 1: cutdir = 0
                else: cutdir = 1

                #---if final depth was reached then end while loop---
                if currentdepth == depth: break;

            #---Savety First---
            code += "G0 Z" + str(thick+zSave) + "\n\n"
                  

        #########################################################################
        #    02.05 end codes                                                    #
        #########################################################################
        
        #---deactivate tool and reset modal codes--- 
        code += gcode.toolOptimizer( self, jIndex, jTools, csvT, "off", True )
        code += gcode.end( self )

        return code 
   

   
    #############################################################################
    #                                                                           #
    #    03.00 determine start-/endtool                                         #
    #                                                                           #
    #############################################################################
    # function returns start/endtool of THIS module for toolOptimizer 
    # "" will be ignored by optimizer 
    # "nT" will force to load no Tool
    def tool( self, joblist, jIndex ):
        
        tool = gcode.findToolchange( self, joblist, jIndex ) 
        return (tool,tool) #(Start, Endtool)

           
   
