'''
#################################################################################
#                                                                               #
#    Lonnox CUT - module Rectangle                                              #
#                                                                               #
#################################################################################
#	C O N T E N T                                  								#
#-------------------------------------------------------------------------------#
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#    02.00 g-code generator                                                     #
#        02.01 local vars                                                       #
#        02.02 mirror module on x-axis                                          #
#        02.03 startcodes                                                       #
#        02.04 g-code generation for module                                     #
#        02.05 end codes                                                        #
#                                                                               #
#    03.00 determine start-/endtool                                             #
#                                                                               #
#################################################################################
'''
__version__ = '1.0'
__license__ = "license.txt"
__author__ = 'Kai Masemann <info@lonnox.de>'

#################################################################################
#                                                                               #
#    00.00 load libraries                                                       #
#                                                                               #
#################################################################################

#---libraries for the layout---
import math
import os
import sys
import gcode
import uni


#################################################################################
#                                                                               #
#    01.00 gloabl variables                                                     #
#                                                                               #
#################################################################################

#---choose/ add section in the module tree---
section    = "Mill 2D"

#---name of the module in the tree---
name       = "Rectangle"
groupEnd   = ""

#---explain picture of the module--- 
picture    = "Rectangle.png"

#---explaining text for helpbutton in lonnox cut---
info = "RectangleInfo"

#---predefined tools---
predefinedTools = False

#---widgets that are displayed to set the parameters---
# T=Text, N=NumberValues, O=OptionList, F=FileButton, C=CheckBox
widget = ("N","N","O","O","N","N","N","N","N","N","N","N","O","O","N")

#---parameter of the module---
labels    = ("Zero Point X", "Zero Point Y", "Orientation","X Y Mode","X","Y",
             "Width (a)","Height (b)","Radius (r)", "Rotation", "Depth","Cutdepth",
             "Cutpath", "Last segment","Feed")

#---option presets for listed parameters---
options   = {"Orientation":      ("Normal","Mirror"),
             "X Y Mode":         ("X & Y Min","Center"),
             "Cutpath":          ("Center","Inside","Outside"),
             "Last segment":     ("Y Min","X Max","Y Max","X Min")}



class module( object ):
    #############################################################################
    #                                                                           #
    #    02.00 g-code generator                                                 #
    #                                                                           #
    #############################################################################
    def gcode( self, jTools, joblist, jIndex, preview=0 ):
        global name
        L = uni.language 
        
        #########################################################################
        #    02.01 local vars                                                   #
        #########################################################################

        #---extrace name, tool and param rows from joblist table---
        jNames = [job[1] for job in joblist]
        jParams = [job[2:] for job in joblist] #for contents, see labels
        
        #---variable values---
        if name in uni.language: LName = uni.language[name]
        else: LName = name
        code1         = "\n(" + ("-"*30) + LName + jNames[jIndex][-4:] + ("-"*30) + ")\n\n"
        code2         = ""
        x0            = jParams[jIndex][0]
        y0            = jParams[jIndex][1]
        orient        = jParams[jIndex][2]
        mode          = jParams[jIndex][3]
        x             = jParams[jIndex][4] # + x0  see rotation calculation                      
        y             = jParams[jIndex][5] # + y0  see rotation calculation                     
        w             = jParams[jIndex][6]                      
        h             = jParams[jIndex][7]                      
        er            = math.fabs( jParams[jIndex][8] )         
        rotation      = jParams[jIndex][9]
        depth         = math.fabs( jParams[jIndex][10] ) 
        cutdepth      = math.fabs( jParams[jIndex][11] ) 
        cutpath       = jParams[jIndex][12]
        lastseg       = jParams[jIndex][13]
        f             = jParams[jIndex][14]                     
        fPlunge       = f * uni.settings[6] / 100         
        mirrorstops   = uni.settings[1]
        zSave         = uni.settings[4]

        #---load tool on,off,m6,g43 and cutterradius---
        csvT = gcode.csvtool( self, jTools[jIndex][0], jTools[jIndex][2] )
        cr   = csvT["cr"]                             
        
        #---load material thickness---
        i = 1
        thick = 0
        while jIndex >= i:
            if jNames[jIndex-i][:-4] == "Rawpart":
                thick = jParams[jIndex-i][7]
                break;
            i += 1

        #---set preview values---
        if preview: depth=1; cutdepth=1;        

        #---escape if essetial values are missing---
        if not depth or not cutdepth or not w or not h: return code1

  
        #########################################################################
        #    02.02 option and variable preparation                              #
        #########################################################################

        #---convert middle x y mode to edge x y mode---
        if mode == "Center":
            x = x - (w / 2)
            y = y - (h / 2)
    
        #---set/toggle cutter compensation value---
        if cutpath == "Inside":
            #---for proper inner path minimum edge radius must be cutter radius---
            if cr > er : er = cr
            cr = -cr 
        if cutpath == "Center": cr = 0

        
        #---build gcode for right moving steps with y min as last segment---
        rcutVal = [[1, x+w-er, y-cr  , f     ],
                   [3, x+w+cr, y+er  , cr+er ],
                   [1, x+w+cr, y+h-er, f     ],
                   [3, x+w-er, y+h+cr, cr+er ],
                   [1, x+er  , y+h+cr, f     ],
                   [3, x-cr  , y+h-er, cr+er ],
                   [1, x-cr  , y+er  , f     ],
                   [3, x+er  , y-cr  , cr+er ]]
        lcutVal = [[1, x+er  , y-cr  , f     ],
                   [2, x-cr  , y+er  , cr+er ],
                   [1, x-cr  , y+h-er, f     ],
                   [2, x+er  , y+h+cr, cr+er ],
                   [1, x+w-er, y+h+cr, f     ],
                   [2, x+w+cr, y+h-er, cr+er ],
                   [1, x+w+cr, y+er  , f     ],
                   [2, x+w-er, y-cr  , cr+er ]]

        #---rotation calculation---
        for i,(vg,vx,vy,vfr) in enumerate(rcutVal):
            rr = gcode.distancePP( self, (0,0), (vx,vy) )
            ra = rotation + gcode.pointAngle360( self, 0,0, vx,vy, metric=1 )
            if ra > 360: ra -= 360
            rcutVal[i][1],rcutVal[i][2] = gcode.circlePoint( self, x0, y0, ra, rr ) 
        for i,(vg,vx,vy,vfr) in enumerate(lcutVal):
            rr = gcode.distancePP( self, (0,0), (vx,vy) )
            ra = rotation + gcode.pointAngle360( self, 0,0, vx,vy, metric=1 )
            if ra > 360: ra -= 360
            lcutVal[i][1],lcutVal[i][2] = gcode.circlePoint( self, x0, y0, ra, rr ) 
            

        #---recalculate mirror values---
        if orient == "Mirror":
            for i in range(8):
                rcutVal[i][1] = mirrorstops-rcutVal[i][1]              
                if rcutVal[i][0] == 3: rcutVal[i][0] = 2              
                lcutVal[i][1] = mirrorstops-lcutVal[i][1]              
                if lcutVal[i][0] == 2: lcutVal[i][0] = 3              

 
        #---build gcode for right/left moving steps with y min as last segment---
        rcut = [] 
        for vg,vx,vy,vfr in rcutVal:
            if vg==1: rcut.append( "G1 X{:.3f} Y{:.3f} F{:.0f}\n".format(vx,vy,vfr) )
            if vg==2: rcut.append( "G2 X{:.3f} Y{:.3f} R{:.3f}\n".format(vx,vy,vfr) )
            if vg==3: rcut.append( "G3 X{:.3f} Y{:.3f} R{:.3f}\n".format(vx,vy,vfr) )
        lcut = []
        for vg,vx,vy,vfr in lcutVal:
            if vg==1: lcut.append( "G1 X{:.3f} Y{:.3f} F{:.0f}\n".format(vx,vy,vfr) )
            if vg==2: lcut.append( "G2 X{:.3f} Y{:.3f} R{:.3f}\n".format(vx,vy,vfr) )
            if vg==3: lcut.append( "G3 X{:.3f} Y{:.3f} R{:.3f}\n".format(vx,vy,vfr) )
                
        #---rotate step lists--- 
        if lastseg == "X Max": lcut = gcode.rotate(self, lcut, 6); rcut = gcode.rotate(self, rcut, 2)
        if lastseg == "Y Max": lcut = gcode.rotate(self, lcut, 4); rcut = gcode.rotate(self, rcut, 4)
        if lastseg == "X Min": lcut = gcode.rotate(self, lcut, 2); rcut = gcode.rotate(self, rcut, 6)

        
        #---check if there is a radius in final path and maby delete G2/G3---
        if (er + cr ) <= 0:
            for i in range(8):
                if i%2: rcut[i] = ""; lcut[i] = ""


        #########################################################################
        #    02.03 startcodes                                                   #
        #########################################################################

        #---add startcode and activate tool---
        code1 += gcode.start( self, thick ) 
        code1 += gcode.toolOptimizer( self, jIndex, jTools, csvT, "on", True )
        
        
        #########################################################################
        #    02.05 g-code generation for module                                 #
        #########################################################################

        #--- rectangle cut: move to (compensated) startposition---
        i = rcut[0].find("F")
        s = rcut[0][0:i]
        code1 += "G0 Z{:.3f}\n".format( thick+zSave ) 
        code1 += s.replace("G1","G0") + "\n"

        #--- last segment cut: move to (compensated) startposition---
        i = lcut[0].find("F")
        s = lcut[0][0:i]
        code2 += "G0 Z{:.3f}\n".format( thick+zSave )
        code2 += s.replace("G1","G0") + "\n"
        
        #---loop until final depth was reached---
        dirtoggle = 0 
        currentdepth = 0.00
        while 1:

            #---set next depth level or final depth---
            currentdepth += cutdepth
            if currentdepth > depth: currentdepth = depth

            #---update depth for rectangle cut and last segment cut---
            code1 += "G1 Z{:.3f} F{:.0f}\n".format( thick-currentdepth, fPlunge )
            code2 += "G1 Z{:.3f} F{:.0f}\n".format( thick-currentdepth, fPlunge )

            #---build steps for left cut direction and last segment---  
            if dirtoggle:
                for row in lcut: code1 += row   
                code2 += lcut[0]  

            #---build steps for right cut direction and last segment---  
            else:
                for row in rcut: code1 += row   
                code2 += rcut[0]  
            dirtoggle ^= 1

            #---if final depth was reached then end while loop---
            if currentdepth >= depth: break;
            
        #---put rectangle cut and last segment cut together---
        code1 += "\n" + code2 + "\n"
        code1 += "G0 Z" + str(thick+zSave) + "\n\n"
                  

        #########################################################################
        #    02.05 end codes                                                    #
        #########################################################################
        
        #---deactivate tool and reset modal codes--- 
        code1 += gcode.toolOptimizer( self, jIndex, jTools, csvT, "off", True )
        code1 += gcode.end( self )

        return code1 
   


    #############################################################################
    #                                                                           #
    #    03.00 determine start-/endtool                                         #
    #                                                                           #
    #############################################################################
    # function returns start/endtool of THIS module for toolOptimizer 
    # "" will be ignored by optimizer 
    # "nT" will force to load no Tool
    def tool( self, joblist, jIndex ):
        
        tool = gcode.findToolchange( self, joblist, jIndex ) 
        return (tool,tool) #(Start, Endtool)

