'''
#################################################################################
#                                                                               #
#    dxf Extend Import Library                                                  #
#                                                                               #
#################################################################################
# Copyright 2022 Kai Masemann                                                   #
#################################################################################
# Notes:
# - dxfFilter.py is handled like modules. Default module is part off the /module/
#   directory, if a dxfFilter file is present in the machine path, then that will
#   be used instead.
#################################################################################
'''
__version__ = '1.0'
__license__ = ""
__author__ = 'Kai Masemann <info@lonnox.de>'

#################################################################################
#                                                                               #
#    00.00 Load Libraries                                                       #
#                                                                               #
#################################################################################

#---Libraries For The Layout---
import math
import os
import sys
import gcode 
import uni
from copy import copy, deepcopy


#################################################################################
#                                                                               #
#    01.00 Gloabl Variables                                                     #
#                                                                               #
#################################################################################

#---File And Dirnames for frozen/ unfrozen application---
if getattr(sys, 'frozen', False):
    # frozen
    path = os.path.dirname(sys.executable) + "/"
else:
    # unfrozen
    path = os.path.dirname(os.path.realpath(__file__)) + "/"
homePath = os.path.expanduser("~") + "/LonnoxCUT/"

#---Default Job Settings---
jS  = [False, 0,[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]] #jobSetup
jSG = [False,+1,[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]] #jobSetup - Group
jSE = [False,-1,[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0]] #jobSetup - GroupEnd



#####################################################################################
#                                                                                   #
#    02.00 Dxf Filter Names
#                                                                                   #
#####################################################################################
# This method will return the filters that are available for dxf imports. The 
# method "filter" will be called with the name of the filter in the parameter
# customFilter. 
# The "TrunCad" filter will be catched in LCUT.py - "Menu Import dxf Files"
# and uses the old TrunCad Dxf Method.
#
# FILTER DESCRIPTION:
# Standard | 0€ |  - | General
# TrunCad | 0€ | - | General
# Auto Tooling | 50€ | VDrill, Holeline | Customer, Tenselieros 
# SmartWop     | 50€ | VDrill, Holeline, Raw, HDrill, Saw, Info | Customer
def getNames():
    #return ["Standard","TrunCad","Auto Tooling","SmartWop"]
    return ["Standard","TrunCad"]
    
    

#####################################################################################
#                                                                                   #
#    03.00 DXF Filters Proceeding
#                                                                                   #
#####################################################################################
# The customFilter parameter will be used to decide how to convert the 
# entities of the dxf file to lonnox cut modules. For example, a series of
# holes can be convertet to multiple circles, drills or to a holeline.
# 
# The customFilters offer the use of useOptions by a LAYER SYNTAX
# getLayeruseOptions()
def filter( joblist,entys,customFilter,filePath ):
    
    #entys.append( {'entity': 'CIRCLE', 'layer': 'Tokkahonga', 'color': '0', 'xyzr': [300.0, 30.0, 0.0, 12.5]} )

    #---Add Key In Entys To Indicate That Entity Is Already Applied To Joblist---
    for i in range(len(entys)):
        entys[i]["applied"] = False

    #---Search In Geometries For Module Content In The Right Order---
    for i in range(len(entys)):
        #---No Filter (Default)--- 
        if customFilter == "Standard":      
            joblist,entys = ellipseToEllipse( joblist, entys, i, lPreset="", aTool=0 )
            joblist,entys = arcToBow( joblist, entys, i, lPreset="", aTool=0 )
            joblist,entys = circleToCircle( joblist, entys, i, lPreset="", aTool=0 )
            joblist,entys = lineToLine( joblist, entys, i, lPreset="", aTool=0 )
            joblist,entys = pointToDrill( joblist, entys, i, lPreset="", aTool=0 )
            joblist,entys = splineToSpline( joblist, entys, i, lPreset="", aTool=0 )
            joblist,entys = polyToContour( joblist, entys, i, lPreset="", aTool=0 )
        elif customFilter == "SmartWop":
            joblist,entys = ellipseToEllipse( joblist, entys, i, lPreset="SmartWop", aTool=1 )
            joblist,entys = arcToBow( joblist, entys, i, lPreset="SmartWop", aTool=1 )
            joblist,entys = circleToHolelineAT( joblist, entys, i, lPreset="SmartWop", aTool=1 )
            joblist,entys = circleToDrillAT( joblist, entys, i, lPreset="SmartWop", aTool=1 )
            joblist,entys = circleToCircle( joblist, entys, i, lPreset="SmartWop", aTool=1 )
            joblist,entys = lineToXSawSW( joblist, entys, i, lPreset="SmartWop", aTool=1 )
            joblist,entys = lineToXHDrillSW( joblist, entys, i, lPreset="SmartWop", aTool=1 )
            joblist,entys = lineToLineSW( joblist, entys, i, lPreset="SmartWop", aTool=1 )
            joblist,entys = pointToDrill( joblist, entys, i, lPreset="SmartWop", aTool=1 )
            joblist,entys = splineToSpline( joblist, entys, i, lPreset="SmartWop", aTool=1 )
            joblist,entys = polyToRawpartSW( joblist, entys, i, lPreset="SmartWop", aTool=1, f=filePath )
            joblist,entys = polyToFormatSW( joblist, entys, i, lPreset="SmartWop", aTool=1 )
            joblist,entys = polyToContour( joblist, entys, i, lPreset="SmartWop", aTool=1 )
        elif customFilter == "Auto Tooling":
            joblist,entys = ellipseToEllipse( joblist, entys, i, lPreset="", aTool=1 )
            joblist,entys = arcToBow( joblist, entys, i, lPreset="", aTool=1 )
            joblist,entys = circleToHolelineAT( joblist, entys, i, lPreset="", aTool=1 )
            joblist,entys = circleToDrillAT( joblist, entys, i, lPreset="", aTool=1 )
            joblist,entys = circleToCircle( joblist, entys, i, lPreset="", aTool=1 )
            joblist,entys = lineToLine( joblist, entys, i, lPreset="", aTool=1 )
            joblist,entys = pointToDrill( joblist, entys, i, lPreset="", aTool=1 )
            joblist,entys = splineToSpline( joblist, entys, i, lPreset="", aTool=1 )
            joblist,entys = polyToContour( joblist, entys, i, lPreset="", aTool=1 )
            
    return joblist

#####################################################################################
#                                                                                   #
#    04.00 Standard Filters
#                                                                                   #
#####################################################################################

#############################################################################
#    04.01 Ellipse To Ellipse                                               #
#############################################################################
def ellipseToEllipse( joblist,entys,index,lPreset,aTool ):
    global jS     
    
    #---Convert Ellipse To Ellipse---
    e = entys[index]
    if e["entity"] == "ELLIPSE" and not e["applied"]:
        p =  {"tool":[],"feed":"2000","depth":"1","cutdepth":"1","cutpath":"Center"}
        #---Choose Placeholder Tool---
        if aTool: p["tool"] = searchToollist( joblist, -1, "Other" )
        #---Get Parameter From Layer---
        if lPreset != "": p = getLayerParameter( p, joblist, lPreset, e["layer"] )
        #---Add Toolchange If Neccessary---
        if toolchangeNeeded( joblist, p["tool"] ): 
            joblist.append( p["tool"] )

        #---Calculate Parameters---
        rotation = math.degrees( math.atan2(-e["xyzMajor"][0],e["xyzMajor"][1]) ) +90
        width = math.sqrt(e["xyzMajor"][0]**2 + e["xyzMajor"][1]**2) * 2 
        height = width * e["ratio"] 
        startAng = (360 / (math.pi*2)) * e["piRange"][0] 
        endAng = (360 / (math.pi*2)) * e["piRange"][1] 
        #---Add Job---
        name = jobName(joblist,"Ellipse")
        joblist.append( [deepcopy(jS),name,'0.00','0.00','Normal',e["xyzCenter"][0],
                         e["xyzCenter"][1],width,height,rotation,startAng,endAng,
                         p["depth"],p["cutdepth"],p["cutpath"],p["feed"]] )
        joblist[-1] = jobFormat(joblist[-1])
        #---Mark As Applied---
        entys[index]["applied"] = True
    
    return joblist,entys


#############################################################################
#    04.02 Arc To Bow                                                       #
#############################################################################
def arcToBow( joblist,entys,index,lPreset,aTool ):
    global jS     
    
    #---Convert Arc To Bow---
    e = entys[index]
    if e["entity"] == "ARC" and not e["applied"]:
        p = {"tool":[],"feed":"2000","depth":"1","cutdepth":"1","cutpath":"Center"}
        #---Choose Placeholder Tool---
        if aTool: p["tool"] = searchToollist( joblist, -1, "Other" )
        #---Get Parameter From Layer---
        if lPreset != "": p = getLayerParameter( p, joblist, lPreset, e["layer"] )
        #---Add Toolchange If Neccessary---
        if toolchangeNeeded( joblist, p["tool"] ): 
            joblist.append( p["tool"] )
  
        #---Add Job---
        name = jobName(joblist,"Bow")
        joblist.append( [deepcopy(jS),name,'0.00','0.00','Normal',e["xyzr"][0],
                         e["xyzr"][1],e["xyzr"][3],
                         e["angleRange"][0],e["angleRange"][1],
                         p["depth"],p["cutdepth"],p["cutpath"],p["feed"]] )
        joblist[-1] = jobFormat(joblist[-1])
        #---Mark As Applied---
        entys[index]["applied"] = True

    return joblist,entys


#############################################################################
#    04.03 Circle To Circle                 
#############################################################################
def circleToCircle( joblist,entys,index,lPreset,aTool ):
    global jS     
    
    #---Convert Circle To Circle---
    e = entys[index]
    if e["entity"] == "CIRCLE" and not e["applied"]:
        p = {"tool":[],"feed":"2000","depth":"10","cutdepth":"5","cutpath":"Center"}
        #---Choose Placeholder Tool---
        if aTool: p["tool"] = searchToollist( joblist, -1, "Other" )
        #---Get Parameter From Layer---
        if lPreset != "": p = getLayerParameter( p, joblist, lPreset, e["layer"] )
        #---Add Toolchange If Neccessary---
        if toolchangeNeeded( joblist, p["tool"] ): 
            joblist.append( p["tool"] )

        #---Add Job---
        name = jobName(joblist,"Circle")
        joblist.append( [deepcopy(jS),name,'0.00','0.00','Normal',e["xyzr"][0],
                         e["xyzr"][1],e["xyzr"][3],
                         p["depth"],p["cutdepth"],p["cutpath"],'Y Min',p["feed"]] )
        joblist[-1] = jobFormat(joblist[-1])
        #---Mark As Applied---
        entys[index]["applied"] = True

    return joblist,entys


#############################################################################
#    04.04 Line To Line                                                     #
#############################################################################
def lineToLine( joblist,entys,index,lPreset,aTool ):
    global jS     
    
    #---Convert Line To Line---
    e = entys[index]
    if e["entity"] == "LINE" and not e["applied"]:
        p = {"tool":[],"cutpath":"Center","depth":"1","cutdepth":"1","feed":"2000"}
        #---Choose Placeholder Tool---
        if aTool: p["tool"] = searchToollist( joblist, -1, "Other" )
        #---Get Parameter From Layer---
        if lPreset != "": p = getLayerParameter( p, joblist, lPreset, e["layer"] )
        #---Add Toolchange If Neccessary---
        if toolchangeNeeded( joblist, p["tool"] ): 
            joblist.append( p["tool"] )

        #---Add Job---
        name = jobName(joblist,"Line")
        joblist.append( [deepcopy(jS),name,'0.00','0.00','Normal',e["xyzStart"][0],
                         e["xyzStart"][1],e["xyzEnd"][0],e["xyzEnd"][1],
                         '0',p["cutpath"],p["depth"],p["cutdepth"],p["feed"]] )
        joblist[-1] = jobFormat(joblist[-1])
        #---Mark As Applied---
        entys[index]["applied"] = True

    return joblist,entys


#############################################################################
#    04.05 Point To Drill                                                   #
#############################################################################
def pointToDrill( joblist,entys,index,lPreset,aTool ):
    global jS     
    
    #---Convert Line To Line---
    e = entys[index]
    if e["entity"] == "POINT" and not e["applied"]:
        p = {"tool":[],"depth":"1","cutdepth":"1","feed":"1200"}
        #---Choose Placeholder Tool---
        if aTool: p["tool"] = searchToollist( joblist, -1, "Other" )
        #---Get Parameter From Layer---
        if lPreset != "": p = getLayerParameter( p, joblist, lPreset, e["layer"] )
        #---Add Toolchange If Neccessary---
        if toolchangeNeeded( joblist, p["tool"] ): 
            joblist.append( p["tool"] )

        #---Add Job---
        name = jobName(joblist,"VerticalDrill")
        joblist.append( [deepcopy(jS),name,'0.00','0.00','Normal',e["xyz"][0],
                         e["xyz"][1],p["depth"],p["cutdepth"],p["feed"]] )
        joblist[-1] = jobFormat(joblist[-1])
        #---Mark As Applied---
        entys[index]["applied"] = True

    return joblist,entys


#############################################################################
#    04.06 Spline To Spline                                                 #
#############################################################################
def splineToSpline( joblist,entys,index,lPreset,aTool ):
    global jS, jSG, jSE     
    
    #---Convert Spline To Spline---
    e = entys[index]
    if e["entity"] == "SPLINE" and not e["applied"]:
        p = {"tool":[],"cutpath":"Center","depth":"1","cutdepth":"1","feed":"1200"}
        #---Choose Placeholder Tool---
        if aTool: p["tool"] = searchToollist( joblist, -1, "Other" )
        #---Get Parameter From Layer---
        if lPreset != "": p = getLayerParameter( p, joblist, lPreset, e["layer"] )
        #---Add Toolchange If Neccessary---
        if toolchangeNeeded( joblist, p["tool"] ): 
            joblist.append( p["tool"] )

        #---Calculate Parameter---
        splType,conType = "Cubic B-Spline","Open"
        if e["degree"] == 2: splType = "Quadratic B-Spline"
        if e["closed"]: conType = "Close"
        #---Add Spline Group---
        name = jobName(joblist,"Spline-Group")
        joblist.append( [deepcopy(jSG),name,'0.00','0.00','Normal',splType,
                         'Control Points','Low',conType,'0.00',
                         p["cutpath"],p["depth"],p["cutdepth"],p["feed"]] )
        #---Add Spline Points---
        for x,y,z,w in e["xyzwControl"]:
            name = jobName(joblist,"Spline-Point")
            joblist.append( [deepcopy(jS),name,'0.00','0.00',x,y,w] )
            joblist[-1] = jobFormat(joblist[-1])
        #---Add Spline End---
        name = jobName(joblist,"Spline-End")
        joblist.append( [deepcopy(jSE),name,'0.00','0.00'] )
        #---Mark As Applied---
        entys[index]["applied"] = True

    return joblist,entys


#############################################################################
#    04.07 Polyline To Contour                                              #
#############################################################################
def polyToContour( joblist,entys,index,lPreset,aTool ):
    global jS, jSG, jSE     
    
    #---Convert Polyline To Contour---
    e = entys[index]
    if e["entity"] == "POLYLINE" and not e["applied"]: 
        p = {"tool":[],"cutpath":"Center","depth":"1","cutdepth":"1","feed":"1200"}
        #---Choose Placeholder Tool---
        if aTool: p["tool"] = searchToollist( joblist, -1, "Other" )
        #---Get Parameter From Layer---
        if lPreset != "": p = getLayerParameter( p, joblist, lPreset, e["layer"] )
        #---Add Toolchange If Neccessary---
        if toolchangeNeeded( joblist, p["tool"] ): 
            joblist.append( p["tool"] )

        #---Only Use Polyline Points---
        xyzb = [(x,y,z,b) for x,y,z,b,c in e["xyzbc"] if not c]
        #---Close Polylines---
        if e["closed"] : xyzb += xyzb[:1]
        #---Add Contour Group---
        name = jobName(joblist,"Contour-Group")
        joblist.append( [deepcopy(jSG),name,'0.00','0.00','Normal','Open',p["cutpath"],
                         p["depth"],p["cutdepth"],p["feed"]] )
        #---Add Contour Points From POLYLINE/LWPOLYLINE---
        xPrev,yPrev,bPrev = 0,0,0
        xList,yList,bList = "","",""
        for j,(x,y,z,b) in enumerate(xyzb):
            #---Insert Bulge Bows---
            if bPrev != 0.0:
                for bx,by,br in gcode.dxfContourRadius( None,(xPrev,yPrev),(x,y),bPrev): 
                    xList += str(round(bx,3)) + ";"
                    yList += str(round(by,3)) + ";"
                    bList += str(round(br,3)) + ";" 
            xPrev,yPrev,bPrev = x,y,b
            #---Add Straight Lines---
            xList += str(round(x,3)) + ";"
            yList += str(round(y,3)) + ";"
            bList += "0.0;"                
        #---Add Contour List Point---            
        name = jobName(joblist,"Contour-List")
        joblist.append( [deepcopy(jS),name,'0.00','0.00',xList,yList,bList] )
        #---Add Contour End---
        name = jobName(joblist,"Contour-End")
        joblist.append( [deepcopy(jSE),name,'0.00','0.00'] )
        #---Mark As Applied---
        entys[index]["applied"] = True

    return joblist,entys



#####################################################################################
#                                                                                   #
#    05.00 Auto Tooling Filters
#                                                                                   #
#####################################################################################

#############################################################################
#    05.01 Circle To Drill (Auto Tooling)  
#############################################################################
def circleToDrillAT( joblist,entys,index,lPreset,aTool ):
    global jS     
    
    #---Convert Circle To Drill---
    e = entys[index]
    if e["entity"] == "CIRCLE" and not e["applied"]:
        p = {"tool":[],"depth":"8","cutdepth":"10","plunge feed":"1200"}
        #---Check For Available Drill Tool---
        if aTool: p["tool"] = searchToollist( joblist, e["xyzr"][3], "Drill bit" )
        #---Get Parameter From Layer---
        if lPreset != "": p = getLayerParameter( p, joblist, lPreset, e["layer"] )
        #---If Tool Available Add/Convert Jobs---
        if p["tool"] != []: 
            #---Add Toolchange If Neccessary---
            if toolchangeNeeded( joblist, p["tool"] ): 
                joblist.append( p["tool"] )
            
            #---Add Drill Job---
            name = jobName( joblist, "VerticalDrill" )
            joblist.append( [deepcopy(jS),name,'0.00','0.00','Normal',e["xyzr"][0],
                             e["xyzr"][1],p["depth"],p["cutdepth"],p["plunge feed"]] )
            joblist[-1] = jobFormat(joblist[-1])
            #---Mark As Applied---
            entys[index]["applied"] = True

    return joblist,entys


#############################################################################
#    05.02 Circle To Holeline (Auto Tooling)
#############################################################################
def circleToHolelineAT( joblist,entys,index,lPreset,aTool ):
    global jS
    
    #---Check If Current Job Is A Circle---
    e = entys[index]
    if e["entity"] == "CIRCLE" and not e["applied"]:
        p = {"tool":[],"depth":"8","plunge feed":"1200"}
        #---Get Parameter From Layer---
        if lPreset != "": p = getLayerParameter( p, joblist, lPreset, e["layer"] )
        
        #---Itterate Through Following Entities---
        cE = [e]
        cI = [index]
        r = 2.5
        for i in range(index+1,len(entys)):
            eY = entys[i]
            #---Filter Circles By Y Level And Radius---
            if eY["entity"] == "CIRCLE" and not eY["applied"]:
                if eY["xyzr"][3] == r and eY["xyzr"][1] == cE[0]["xyzr"][1]:
                    #---Add Dirllholes On Equal Y Level---
                    cE.append(eY)
                    cI.append(i)
        
        #---Filter Circles By 32mm Grid---
        cEX = [ eX["xyzr"][0] for eX in cE ] 
        dE = [e]
        dI = [index]
        while 1:
            xPrev,xNext = dE[0]["xyzr"][0] - 32.0, dE[-1]["xyzr"][0] + 32.0
            #---Search For Drills In Positive Direction---
            if round(xPrev,3) in cEX: 
                i = cEX.index(round(xPrev,3))
                dE.insert(0,cE[i])
                dI.insert(0,cI[i])
            #---Search For Drills In Negative Direction---
            elif round(xNext,3) in cEX: 
                i = cEX.index(round(xNext,3))
                dE.append(cE[i])
                dI.append(cI[i])
            #---If No More Drills Are Found Escape Searching---
            else: break; 
            
        #---Create HoleLine If Possible---
        if len(dE) >= int(uni.settings[7]) and int(uni.settings[7]) > 1:
            #---Add Holeline Job---
            name = jobName(joblist,"HoleLine-5mm")
            joblist.append( [deepcopy(jS),name,'0.00','0.00','Normal',
                             dE[0]["xyzr"][0],dE[-1]["xyzr"][0],
                             dE[0]["xyzr"][1],'0.00','0.00','0.00',
                             '0.00','0.00',p["depth"],p["plunge feed"]] )
            joblist[-1] = jobFormat(joblist[-1])
            #---Mark As Applied If Holeline Is Possible---
            for i in dI: entys[i]["applied"] = True
         
    return joblist,entys



#####################################################################################
#                                                                                   #
#    06.00 SmartWop Filters
#                                                                                   #
#####################################################################################

#############################################################################
#    06.01 Polyline To Rawpart (SmartWop)                                   #
#############################################################################
def polyToRawpartSW( joblist,entys,index,lPreset,aTool,f ):
    global jS
    
    #---Convert Polyline To Rawpart---
    e = entys[index]
    if e["entity"] == "POLYLINE" and e["layer"].startswith("Werkstk") and not e["applied"]:
        p = {"thickness":"19.0","info":""}
        #---Get Parameter From Layer---
        if lPreset != "": p = getLayerParameter( p, joblist, lPreset, e["layer"] )

        #---Extract Bounds---
        xMin = min( [x for x,y,z,b,c in e["xyzbc"]] )
        yMin = min( [y for x,y,z,b,c in e["xyzbc"]] )
        xMax = max( [x for x,y,z,b,c in e["xyzbc"]] )
        yMax = max( [y for x,y,z,b,c in e["xyzbc"]] )
        #---Add Jobs If Possible---
        if xMin < xMax and yMin < yMax: 
            #---Add Part Info Job---
            name = jobName(joblist,"Part info")
            joblist.insert( 0,[deepcopy(jS),name,'','','',p["info"],'',
                               xMax-xMin,yMax-yMin,p["thickness"],'','','','',f] )
            joblist[0] = jobFormat(joblist[0])
            #---Add Rawpart Job---
            name = jobName(joblist,"Rawpart")
            joblist.insert( 1,[deepcopy(jS),name,'0.00','0.00','Normal',
                               xMin,yMin,xMax-xMin,yMax-yMin,p["thickness"]] )
            joblist[1] = jobFormat(joblist[1])
            #---Mark As Applied---
            entys[index]["applied"] = True

    return joblist,entys


#############################################################################
#    06.02 Polyline To Format (SmartWop)
#############################################################################
def polyToFormatSW( joblist,entys,index,lPreset,aTool ):
    global jS
    
    #---Convert Polyline To Format---
    e = entys[index]
    if e["entity"] == "POLYLINE" and e["layer"].startswith("V_Fraes") and not e["applied"]:
        p = {"tool":[],"feed":"2000"}
        #---Choose First Endmill Tool In List---
        if aTool: p["tool"] = searchToollist( joblist, -1, "Endmill" )
        #---Get Parameter From Layer---
        if lPreset != "": p = getLayerParameter( p, joblist, lPreset, e["layer"] )

        #---Extract Bounds---
        xMin = min( [x for x,y,z,b,c in e["xyzbc"]] )
        yMin = min( [y for x,y,z,b,c in e["xyzbc"]] )
        xMax = max( [x for x,y,z,b,c in e["xyzbc"]] )
        yMax = max( [y for x,y,z,b,c in e["xyzbc"]] )
        #---Add Jobs If Possible---
        if xMin < xMax and yMin < yMax: 
            #---Insert After Last Rawpart Or On The End---
            found = False
            for i in range(len(joblist)-1,-1,-1):
                if joblist[i][1].startswith("Rawpart"):
                    found = True
                    i += 1
                    break;                    
            if not found: i = len(joblist)                    
            #---Add Toolchange If Neccessary---
            if toolchangeNeeded( joblist[:i], p["tool"] ): 
                joblist.insert( i,p["tool"] )
                i += 1
            #---Add Format Job---
            name = jobName(joblist,"Format")
            joblist.insert( i,[deepcopy(jS),name,'0.00','0.00','Normal',
                               xMin,yMin,xMax-xMin,yMax-yMin,p["feed"]] )
            joblist[i] = jobFormat(joblist[i])
            #---Mark As Applied---
            entys[index]["applied"] = True

    return joblist,entys


#############################################################################
#    06.03 Line To Line (SmartWop)
#############################################################################
def lineToLineSW( joblist,entys,index,lPreset,aTool ):
    global jS     

    #---Ignore Lines From Layer For Horizontal Drills---
    e = entys[index]
    if e["entity"] == "LINE" and e["layer"] != "0":
        joblist,entys = lineToLine( joblist,entys,index,lPreset,aTool )
    return joblist,entys


#############################################################################
#    06.04 Polyline To Groovesaw X (SmartWop)      
#############################################################################
def lineToXSawSW( joblist,entys,index,lPreset,aTool ):
    global jS
    
    #---Convert Polyline To Groovesaw---
    e = entys[index]
    if e["entity"] == "LINE" and e["layer"].startswith("V_Saeg") and not e["applied"]:
        p = {"direction":"Climb milling","depth":"8","width":"4","blade":"4","feed":"5000"}
        #---Choose Placeholder Tool---
        if aTool: p["tool"] = searchToollist( joblist, -1, "Other" )
        #---Get Parameter From Layer---
        if lPreset != "": p = getLayerParameter( p, joblist, lPreset, e["layer"] )

        #---Search For Second Line Of Group--- 
        found = False
        for index2 in range(index+1,len(entys)):
            e2 = entys[index2]
            #---Filter Circles By Same X Start And End Values---
            if e2["entity"] == "LINE" and e2["layer"] == e["layer"] and not e2["applied"]:
                if e2["xyzStart"][0] == e["xyzStart"][0] and e2["xyzEnd"][0] == e["xyzEnd"][0]:
                    found = True
                    break
        if found:
            #---Extract Groove Bounds---
            xMin = min( [e["xyzStart"][0],e["xyzEnd"][0]] )
            xMax = max( [e["xyzStart"][0],e["xyzEnd"][0]] )
            yMin = min( [e["xyzStart"][1],e2["xyzStart"][1]] )
            yMax = max( [e["xyzStart"][1],e2["xyzStart"][1]] )
            #---Add Groovesaw Job---
            name = jobName(joblist,"Groovesaw X")
            joblist.append( [deepcopy(jS),name,'0.00','0.00','Normal',
                             p["direction"],xMin,xMax,yMin,p["depth"],
                             yMax-yMin,p["blade"],p["feed"]] )
            joblist[-1] = jobFormat(joblist[-1])
            #---Mark As Applied---
            entys[index ]["applied"] = True
            entys[index2]["applied"] = True
            
    return joblist,entys


#############################################################################
#    06.05 Line To X Horizontal Drill (SmartWop)                                    
#############################################################################
def lineToXHDrillSW( joblist,entys,index,lPreset,aTool ):

    #---Convert Lines To Horizontal X Drill---
    e = entys[index]
    if e["entity"] == "LINE" and e["layer"] == "0" and not e["applied"]:
        p = {"x":"0","y":"0","z":"0","depth":"0","direction":"","plunge feed":"1200"}
        #---Get Parameter From Layer---
        if lPreset != "": p = getLayerParameter( p, joblist, lPreset, e["layer"] )

        #---Search For XMin And XMax Of Last Rawpart---
        rawMin, rawMax = 0.0, 0.0
        for i in range(len(joblist)-1,-1,-1):
            if joblist[i][1].startswith("Rawpart"):
                rawMin = float( joblist[i][5] )
                rawMax = rawMin + float( joblist[i][7] )  
                break;                    
        
        #---Check If Rawpart Was Found---
        if rawMax-rawMin > 0.0:
            #--- And Line Entity Is On The Rawpart Edge---
            if e["xyzStart"][0] == e["xyzEnd"][0] == rawMin:
                p["x"] = str(rawMin)
                p["direction"] = "X positive"
            if e["xyzStart"][0] == e["xyzEnd"][0] == rawMax:
                p["x"] = str(rawMax)
                p["direction"] = "X negative"
            #---Check If Line Is On The Edge---  
            if p["direction"] != "":
                #---Check If There Are 7 More Lines (SmartWop HDrill Geometrie)---
                if index+7 <= len(entys):
                    #---Check If Line Is Part Of A Rectangle---                
                    if( (entys[index+0]["xyzStart"] == entys[index+1]["xyzEnd"]) and
                        (entys[index+1]["xyzStart"] == entys[index+2]["xyzEnd"]) and
                        (entys[index+2]["xyzStart"] == entys[index+5]["xyzEnd"]) and
                        (entys[index+5]["xyzStart"] == entys[index+0]["xyzEnd"]) ):
                        #---Add Job---
                        p["depth"] = math.fabs( entys[index+1]["xyzStart"][0] -                        
                                                entys[index+1]["xyzEnd"][0] )
                        yMin = min( [e["xyzStart"][1],e["xyzEnd"][1]] )
                        yMax = max( [e["xyzStart"][1],e["xyzEnd"][1]] )
                        p["y"] = yMin + ( (yMax - yMin) / 2 )
                        p["z"] = e["xyzStart"][2]
                        name = jobName(joblist,"HorizontalDrill")
                        joblist.append( [deepcopy(jS),name,'0.00','0.00','Normal',
                                         p["direction"],p["x"],p["y"],'0','0','0','0','0',
                                         p["z"],p["depth"],p["plunge feed"]] )
                        joblist[-1] = jobFormat(joblist[-1])
                        #---Mark As Applied---
                        print("--------------------------")
                        for i in range(7):
                            entys[index+i]["applied"] = True
                            print("h enty",entys[index+i])                    
    return joblist,entys



#####################################################################################
#                                                                                   #
#    50.00 Helper Methods
#                                                                                   #
#####################################################################################

#############################################################################
#    50.01 Find Available Jobname                                           #
#############################################################################
def jobName( joblist,name ):

    #---Search Jobnumber---
    jobNames = [job[1] for job in joblist]
    no = 1
    while "{} {:03d}".format(name,no) in jobNames: no+=1

    return "{} {:03d}".format(name,no)
    

#############################################################################
#    50.02 Format Job Number Values                                         #
#############################################################################
def jobFormat( job ):
    #---Convert Number Parameters To String---
    for i,para in enumerate(job):
        if isinstance(para,float) or isinstance(para,int):
            job[i] = "{:.3f}".format( job[i] )
    
    return job


#############################################################################
#    50.03 Search For Tool In Toollist
#############################################################################
# - for cutter types see tools.csv
# - raduis -1 = ignore radius
def searchToollist( joblist, radius, cutterType ):
    global jS     
    
    #---Search For Radius And Cutter Type In Tool List---
    tJob = []
    tFound = False
    for tool in uni.toollist:
        #---Check Cutter Type And Raduis If Available---
        try:
            if tool[8] == cutterType:
                if float(tool[6]) == radius or radius == -1:
                    #---Prepare Toolchange Job---
                    name = jobName(joblist,"Toolchange")
                    tJob = [deepcopy(jS),name,'','',tool[0],tool[1]] 
                    tFound = True
                    break;
        except ValueError: pass    
            
    return tJob

    

#############################################################################
#    50.04 Check For Neccessary Toolchange                                  #
#############################################################################
def toolchangeNeeded( joblist, tJob ):
    global jS     
    
    #---Search For Last Toolchange---
    tNeed = True 
    if tJob != []:
        for job in reversed(joblist):
            if job[1].startswith("Toolchange"):    
                #---Check If Toolchange Is Not Needed---
                if job[4] == tJob[4]: tNeed = False
                break;
    #---If No Tool Is Choosen No Toolchange Is Needed---
    else: tNeed = False 

    return tNeed

    
    
#############################################################################
#    50.05 Filter Out Layers                                                #
#############################################################################
def filterOutLayers( entys,index,layers ):
    #---Mark Layers As Applied---
    if entys[index]["layer"] in layers:
        entys[index]["applied"] = True 

    return entys       


#############################################################################
#    50.06 Get Layer useOptions                                                #
#############################################################################
# Toollist Entry: ['7','10mm','M106 P7','M100','      ','G43 H7','5.0','z-','Drill bit','drill.png']
#############################################################################
# _tool20       = Insert toolChange Job with tool 20
# _thickness9.0 = Apply thickness to rawpart
# _feed1000     = Use feedrate of 1000 in job
# _depth10.0    = Use depth of 10.0
# _cutdepth5.0  = Use cut depth of 5.0
# _cutInner     = Choose inner cut path
# _cutOuter     = Choose outer cut path
# _cutLeft      = Choose left cut path
# _cutRight     = Choose right cut path
# _hDrillX+     = Do horizontal drill in positive direction (Only Point Entity)
# _hDrillX-     = Do horizontal drill in negative direction (Only Point Entity)
# _hDrillY+     = Do horizontal drill in y positive direction (Only Point Entity)
# _hDrillY-     = Do horizontal drill in y negative direction (Only Point Entity)
# _sawBlade4.0  = Use saw blade width of 4.0
# _sawDir+      = Use climb milling for groovesaw
# _sawDir-      = Use up-cut milling for groovesaw
def getLayerParameter( parameter,joblist,lPreset,layer ):
    global jS
    
    return parameter
    #Usable For Future Purpose
    '''
    #---Search For useOptions In Layers---
    useOptions =  {"tool":"","feed":"1","depth":"1","cutdepth":"1",
                "cutpath":"Center","cutside":"Left",
                "hDrillX":"X positive","hDrillY":"Y positive","sawBlade":"4",
                "sawDir":"Climb milling"}
    for l in layer.split("_"):
        #---Toolchange---
        if l.upper().startswith("TOOL"):
            try: 
                t = [tool[0] for tool in uni.toollist]
                if l[4:] in t:
                    float( l[4:] )
                    name = jobName(joblist,"Toolchange")
                    useOptions["tool"] = ( [deepcopy(jS),name,'','',l[4:],uni.toollist[t.index(l[4:])][1]] )
            except (IndexError,ValueError): pass 
        #---Thickness---
        elif l.upper().startswith("THICKNESS"):        
            try: 
                float( l[9:] )
                useOptions["thickness"] = l[9:]
            except (IndexError,ValueError): pass
        #---Feedrate---
        elif l.upper().startswith("FEED"):        
            try: 
                float( l[4:] )
                useOptions["feed"] = l[4:]
            except (IndexError,ValueError): pass
        #---Depth---
        elif l.upper().startswith("DEPTH"):        
            try: 
                float( l[5:] )
                useOptions["depth"] = l[5:]
            except (IndexError,ValueError): pass
        #---Cutdepth---
        elif l.upper().startswith("CUTDEPTH"):        
            try: 
                float( l[8:] )
                useOptions["cutdepth"] = l[8:]
            except (IndexError,ValueError): pass
        #---Cutpath---
        elif l.upper().startswith("CUTINNER"): useOptions["cutpath"] = "Inside"
        elif l.upper().startswith("CUTOUTER"): useOptions["cutpath"] = "Outside"
        elif l.upper().startswith("CUTLEFT"): useOptions["cutside"] = "Left"
        elif l.upper().startswith("CUTRIGHT"): useOptions["cutside"] = "Right"
        #---Horizontal Drill Direction---
        elif l.upper().startswith("HDRILLX+"): useOptions["hDrillX"] = "X positive"
        elif l.upper().startswith("HDRILLX-"): useOptions["hDrillX"] = "X negative"
        elif l.upper().startswith("HDRILLY+"): useOptions["hDrillY"] = "Y positive"
        elif l.upper().startswith("HDRILLY-"): useOptions["hDrillY"] = "Y negative"
        #---Saw Blade---
        elif l.upper().startswith("SAWBLADE"):        
            try: 
                float( l[8:] )
                useOptions["sawBlade"] = l[8:]
            except (IndexError,ValueError): pass
        #---Saw Direction---
        elif l.upper().startswith("SAWDIR+"): useOptions["sawDir"] = "Climb milling"
        elif l.upper().startswith("SAWDIR-"): useOptions["sawDir"] = "Up-cut milling"
    return useOptions       
    '''
#    