'''
#################################################################################
#                                                                               #
#    Font Vector Library                                                        #
#                                                                               #
#################################################################################
# Copyright 2020 Kai Masemann                                                   #
#################################################################################
# This file is part of Lonnox CUT                                               #
#                                                                               #
#   Lonnox CUT is free software: you can redistribute it and/or modify          #
#   it under the terms of the GNU General Public License as published by        #
#   the Free Software Foundation, either version 3 of the License, or           #
#   (at your option) any later version.                                         #
#                                                                               #
#   Lonnox CUT is distributed in the hope that it will be useful,               #
#   but WITHOUT ANY WARRANTY; without even the implied warranty of              #
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               #
#   GNU General Public License for more details.                                #
#                                                                               #
#   You should have received a copy of the GNU General Public License           #
#   along with Lonnox CUT.  If not, see <http://www.gnu.org/licenses/>.         #
#                                                                               #
#################################################################################
#	I N H A L T								                                    #
#-------------------------------------------------------------------------------#
#                                                                               #
#    00.00 Load Librarys                                                        #
#                                                                               #
#    01.00 Global Variables                                                     #
#                                                                               #
#    02.00 read font vector data                                                #
#       01 open file contents                                                   #
#       02 prepare data for glyph indexing                                      #
#       03 determine loca table index with cmap table                           #
#       04 read glyph vector data                                               #
#       04.01 glyphs load glyph horizontal metrics                              #
#          02 no outline glyphs (space)                                         #
#          03 simple glyphs                                                     #
#          04 composite glyphs                                                  #
#                                                                               #
#    03.00 get glyph offsets and lenght                                         #
#    03.01 determine glyph offset with loca table                               #
#                                                                               #
#    04.00 spline interpolation                                                 #
#                                                                               #
#################################################################################
'''
__version__ = '1.0'
__license__ = "license.txt"
__author__ = 'Kai Masemann <k.masemann@lavalu.de>'

#################################################################################
#                                                                               #
#    00.00 Load Libraries                                                       #
#                                                                               #
#################################################################################

#---Libraries For The Layout---
import math
import os
import sys
import gcode
from copy import copy, deepcopy


#################################################################################
#                                                                               #
#    01.00 Gloabl Variables                                                     #
#                                                                               #
#################################################################################



#############################################################################
#                                                                           #
#   02.00 read font vector data                                             #
#                                                                           #
#############################################################################
# The Function return the following 2 data arrays (dictonaries):            #
# [0]: tDetail: Details like text y Min/Max values for scaling              #
# [1]: glyphs: glyph array with infos and coordinates of contours           #
#                                                                           #       
# tDetail["yMin"]]: Minimum y Value for all chars in the font               #       
# tDetail["yMax"]]: Maximum y Value for all chars in the font               #       
#                                                                           #       
# glyphs[]: holds the glyph data for every given char from textline         #
# glyphs[]["contours"][]: holds an array with the contours of a glyph.      #
#                         Missing start/endpoints for quadratic bezier      #
#                         curves are added already, as well as the last     #
#                         point with the coords of the first to close the   #
#                         contours. 
#         ["contours"][][]: holds an array with the coordinates of a        #
#                           contour point with the following data:          #
#                           [x,y,controlPointFlag]                          #
#         ["xMin"]: holds the minimum x value of the glyphs bounding box    #
#         ["yMin"]: holds the minimum y value of the glyphs bounding box    #
#         ["xMax"]: holds the maximum x value of the glyphs bounding box    #
#         ["yMax"]: holds the maximum y value of the glyphs bounding box    #
#         ["width"]: holds the distance from  the current glyphs X0         # 
#                    to the next glyphs X0.                                 #
# controlPointFlag: The flag indicates if a contour point is on or off the  #
#                   the contour. Off the contour means that the x and y     #
#                   coordinates of the point describe a control point of a  #
#                   quadratic bezier curve. thats what fonts are made of.   #
#############################################################################
def read( self, textLine, ttfFile ):

    #########################################################################
    #    02.01 open file contents                                           #
    #########################################################################
    # O = Offset; L = Length; A = Array; I = Index; T = Table
    # F = Format; G = Glyph;         
    #---open file---
    if (ttfFile == ""): return [] 
    
    with open(ttfFile, "rb") as f:
        #---read the amount of tables---
        f.seek(4)
        TAmount = int.from_bytes( f.read(2),"big", signed=False )
        
        #---read data of required tables---
        tags = (b'glyf',b'maxp',b'loca',b'cmap',b'hmtx',b'head',b'hhea')
        Ts = {}
        f.seek(12)
        for i in range(TAmount):
            f.seek(12+(i*16))
            tagr = f.read(4)
            for tag in tags:
                if tagr == tag: 
                    f.read(4)
                    O = int.from_bytes( f.read(4),"big" )
                    Ts[str(tag)[2:-1]] = O

        if not "glyf" in Ts: return []            


        #####################################################################
        #    02.02 prepare data for glyph indexing                          #
        #####################################################################

        #---collect necesarry data for glyph point extraction---
        f.seek( Ts["maxp"]+4)
        numG = int.from_bytes( f.read(2),"big" )
        f.seek( Ts["head"]+50)
        locaF = int.from_bytes( f.read(2),"big", signed=1 )
        f.seek( Ts["cmap"]+8)
        cmapMapO = int.from_bytes( f.read(4),"big" )
        f.seek( Ts["cmap"]+cmapMapO )
        cmapF = int.from_bytes( f.read(2),"big" )
        f.seek( Ts["hhea"]+34 )
        numHMetrics = int.from_bytes( f.read(2),"big" )

        #---convert string to byte array---
        chars = []
        charIs = []
        for c in textLine:
            chars.append( int.from_bytes( c.encode("iso-8859-15"),"big" ) )
            
        #####################################################################
        #    02.03 determine loca table index with cmap table               #
        #####################################################################

        #---FORMAT0: determine loca table index with given chars---
        if cmapF == 0:
            cmapAO = Ts["cmap"] + cmapMapO + 6
            for index, char in enumerate(chars):
                if char > 255: continue
                f.seek( cmapAO + char )
                charIs.append( int.from_bytes(f.read(1),"big") )
            
        #---FORMAT4: determine loca table index with given chars---
        elif cmapF == 4:
            #---collect necesarry values for cmap array read---
            f.seek( Ts["cmap"] + cmapMapO + 6)
            segCount = int( int.from_bytes( f.read(2), "big" ) / 2 ) 
            searchRange =  int.from_bytes( f.read(2), "big" )  
            entrySelector = int.from_bytes( f.read(2), "big" )  
            rangeShift = int.from_bytes( f.read(2), "big" ) 
            endCounts = [ int.from_bytes( f.read(2), "big" ) for i in range(segCount) ]
            f.read(2)
            startCounts = [ int.from_bytes( f.read(2), "big" ) for i in range(segCount) ]
            idDeltas = [ int.from_bytes( f.read(2), "big", signed=1 ) for i in range(segCount) ]
            rangeAdr = f.tell()
            idRangeOs = [ int.from_bytes( f.read(2), "big" ) for i in range(segCount) ]
                
            #---calculate loca table index for chars---
            for c in chars:
                for i in range(segCount):
                    if (c <= endCounts[i]) and (c >= startCounts[i]):
                        idD = idDeltas[i] 
                        sC = startCounts[i]
                        idR = idRangeOs[i]
                        if idR == 0:
                            charIs.append( idD + c ) 
                            break;
                        else: 
                            idAdr = idR + (2 * ( c-sC )) + (rangeAdr+(i*2))                            
                            f.seek( int(idAdr) )
                            id = int.from_bytes( f.read(2), "big" )
                            if id > 0:
                                charIs.append( idD + id )       
                            break;

        #---map char indexes to offsets in glyph table---  
        charIOLs = getGlyphIOL( self, f, locaF, Ts["loca"], charIs )
        compData = [0,0,0,0,0]  
        charIOLCs = [[val[0],val[1],val[2],compData] for val in charIOLs] 


        #####################################################################
        #    02.04 read glyph vector data                                   #
        #####################################################################
         
        #---read glyph coords---
        glyphs = []
        textYMin = 0
        textYMax = 0
        for index,charIOLC in enumerate(charIOLCs):
            f.seek( Ts["glyf"] + charIOLC[1] )
            g = {}
            numC = int.from_bytes( f.read(2), "big", signed=1 )   
            g["xMin"] = min = int.from_bytes( f.read(2), "big", signed=1 ) 
            g["yMin"] = int.from_bytes( f.read(2), "big", signed=1 ) 
            g["xMax"] = int.from_bytes( f.read(2), "big", signed=1 ) 
            g["yMax"] = int.from_bytes( f.read(2), "big", signed=1 )  
            gI = charIOLC[0]  
            
            
            #################################################################
            #    02.04.01 glyphs load glyph horizontal metrics              #
            #################################################################
            
            #---if use_my_metrics flag of a composite flag is set use its aw/lsb---
            if charIOLC[3][1]: gI = charIOLC[3][4]
            else: gI = charIOLC[0]  

            #---calculate hmtx table offset---
            fileO = f.tell()
            if gI <= numHMetrics:
                f.seek( Ts["hmtx"] + (gI*4) )
                g["width"] = int.from_bytes( f.read(2), "big", signed=0 )   
                left = int.from_bytes( f.read(2), "big", signed=1 ) 
            else:
                f.seek( Ts["hmtx"] + (numHMetrics*4) )
                g["width"] = int.from_bytes( f.read(2), "big", signed=0 )   
                f.seek( Ts["hmtx"] + (numHMetrics*4) + ((gI-numHMetrics)*2) )
                left = int.from_bytes( f.read(2), "big", signed=1 )  
            
            #---restore file offset--- 
            f.seek(fileO)


            #################################################################
            #    02.04.02 no outline glyphs (space)                         #
            #################################################################

            #---n. o. glyphs point to next glyph with zero length---
            if not charIOLC[2]:
                g["contours"] = []
                glyphs.append( g ) 
                continue
                
            
            #################################################################
            #    02.04.03 simple glyphs                                     #
            #################################################################

            #---simple glyphs---
            if numC > 0:
                ePts = [ int.from_bytes(f.read(2),"big") for i in range(numC) ] 
                insL = int.from_bytes( f.read(2), "big" )  
                f.seek( f.tell()+insL ) 
                flags = []
                #---collect flags for each coordinate---
                while len(flags)-1 < ePts[-1]:
                    flags.append( int.from_bytes(f.read(1),"big") )
                    if (flags[-1] & 0b1000) != 0:                        
                        repeat = int.from_bytes(f.read(1),"big")
                        for i in range(repeat): 
                            flags.append( flags[-1] )
                #---collect control Point Flags and x/y coordinates---
                cPoint = []
                xs = []                    
                for flag in flags:
                    cPoint.append( not (flag & 0b1) )
                    if   (flag & 0b10010) == 0b10000:
                        xs.append( 0 ) 
                    elif (flag & 0b10010) == 0b00010:
                        xs.append( -(int.from_bytes(f.read(1),"big")) )
                    elif (flag & 0b10010) == 0b10010:                            
                        xs.append( int.from_bytes(f.read(1),"big") )
                    elif (flag & 0b10010) == 0b00000:                            
                        xs.append( int.from_bytes(f.read(2),"big",signed=1) )
                ys = []                    
                for flag in flags:
                    if   (flag & 0b100100) == 0b100000:
                        ys.append( 0 ) 
                    elif (flag & 0b100100) == 0b000100:
                        ys.append( -(int.from_bytes(f.read(1),"big")) )
                    elif (flag & 0b100100) == 0b100100:                            
                        ys.append( int.from_bytes(f.read(1),"big") )
                    elif (flag & 0b100100) == 0b000000:                            
                        ys.append( int.from_bytes(f.read(2),"big",signed=1) )
                #---create countour array with missing points and absolute values---
                g["contours"] = []
                contour = [ ]
                c = 0 
                j = 0
                x = min - left + charIOLC[3][2]  
                y = charIOLC[3][3] 
                for i in range(len(flags)):
                    #---add new point between two quad spline control points---
                    if cPoint[i] and c:
                        nPx = x + (xs[i] / 2)
                        nPy = y + (ys[i] / 2)
                        contour.append( [nPx, nPy, 0] )
                    c = cPoint[i]
                    #---add points from glyf table---                         
                    x += xs[i]; y += ys[i]
                    contour.append( [ x, y, cPoint[i] ] )                    
                    #---determine y min/max for of the current text---
                    if y < textYMin: textYMin = y
                    elif y > textYMax: textYMax = y
                    #---split contours---
                    if i == ePts[j]:
                        #---if first point is a control point rotate---
                        if contour[0][2]: contour.append(contour.pop(0))  
                        #---close contour and save in g---
                        contour.append( list(contour[0]) ) 
                        g["contours"].append( list(contour) )
                        contour[:] = []
                        j += 1
                        c = 0

                #---if AddToPrevGlyph Flag is set add contour to last else append--- 
                if charIOLC[3][0]:
                    for cont in g["contours"]: 
                        glyphs[-1]["contours"].append( list(cont) )  
                else: glyphs.append( g ) 

                        
            #################################################################
            #    02.04.04 composite glyphs                                  #
            #################################################################

            #---composite glyphs---
            else:
                #---repeat for every composite glyph component---
                numComp = 0
                AddToPrevGlyph = 0
                flag = 1 << 5
                while (flag & 1 << 5) == 1 << 5:
                    flag = int.from_bytes( f.read(2), "big" )
                    charI = int.from_bytes( f.read(2), "big" )
                    #---flag 0 arg1 and 2 are words---
                    if (flag & 1 << 0) == 1 << 0: 
                        x = int.from_bytes( f.read(2), "big", signed=1 )
                        y = int.from_bytes( f.read(2), "big", signed=1 )
                        y = 0
                    else: 
                        x = int.from_bytes( f.read(1), "big" )
                        y = int.from_bytes( f.read(1), "big" )
                        y = 0
                    #---flag 1 args are xy values---
                    if (flag & 1 << 1) == 0: print("composite: 'point offset mode' not supported")
                    #---flag 3 we have a scale---
                    if (flag & 1 << 3) == 1 << 3: 
                        trash = f.read(2)
                        print("composite: 'simple scale' not supported")
                    #---flag 6 we have an xy scale---
                    elif (flag & 1 << 6) == 1 << 6:
                        trash = f.read(4)
                        print("composite: 'xy scale' not supported")
                    #---flag 7 we have a 2 by 2---
                    elif (flag & 1 << 7) == 1 << 7: 
                        trash = f.read(8)
                        print("compostie: '2by2 scale' not supported")
                    #---flag 9 use my metrics---
                    if (flag & 1 << 9) == 1 << 9: myMetrics = 1
                    else: myMetrics = 0
                    #---add new glyph to charOLXYs table--- 
                    fileO = f.tell()
                    numComp += 1
                    compData = [AddToPrevGlyph, 
                                myMetrics, 
                                x, 
                                y,
                                charI] 
                    compIOLs = getGlyphIOL( self, f, locaF, Ts["loca"], [charI] )
                    charIOLCs.insert( index+numComp , compIOLs[0] + [compData] )
                    f.seek( fileO )
                    AddToPrevGlyph = 1                 
                    
    return ({"yMin":textYMin, "yMax":textYMax},glyphs)     

    

#############################################################################
#                                                                           #
#   03.00 get glyph offsets and lenght                                      #
#                                                                           #
#############################################################################
def getGlyphIOL( self, file, locaFormat, locaAdress, charIndexTable):

    #####################################################################
    #    03.01 determine glyph offset with loca table                   #
    #####################################################################
        
    #---determine glyph offsets with loca table---
    charIOLs = []
    if locaFormat == 0: b = 2; fac = 2
    else: b = 4; fac = 1
    for charI in charIndexTable:
        file.seek( int( locaAdress + (charI*b) ) )
        O = int.from_bytes( file.read(b), "big" ) * fac  
        ONext = int.from_bytes( file.read(b), "big" ) * fac  
        L = ONext - O
        charIOLs.append( [charI, O, L] )
     
    return charIOLs     

	
#############################################################################
#                                                                           #
#   04.00 contur sort from inner to outer contour                           #
#                                                                           #
#############################################################################
# need a well interpolated glyph contour without quad bezier elements
def contSort( self, glyphs ):

    #---search highest point distance of every contour---
    order = []
    for glyph in glyphs:
        for index,cont in enumerate(glyph["contours"]):
            dmax = 0
            for p in cont:
                d = math.fabs( p[0]**2 + p[1]**2 )
                if d > dmax: dmax = d
            order.append( dmax )                
        glyph2 = {"contours":[]}
        for d in order:
            i = order.index(max(order))
            glyph2["contours"].insert( 0, deepcopy(glyph["contours"][i]) )            
            order[i] = 0
        
            
    return
	
	
	
